# Server-sent events (SSE)

**Server-sent events (SSE)** is a technology enabling a browser to receive automatic updates from a server via HTTP connection. The Server-Sent Events EventSource API is standardized as part of HTML5 by the W3C. [Wikipedia](https://en.wikipedia.org/wiki/Server-sent_events)

PeepSo shipped with SSE as an **advanced feature** which, when enabled, replace long-polling method which currently used to update the browser if new data from the server is available, for example, are notification and chat. To enable this feature, a setup needed on the server.

### Technical Requirements
There are some requirements to enable this feature which described here:
  - **Node.js**  
    The server needs to have Node.js installed to run the SSE endpoint. Info regarding Node.js installation can be found [here](https://nodejs.org/en/).
  
  - **Open port**  
    To avoid conflict with Apache, Node.js SSE endpoint listen to a different port. By default it listens to port 8000, however, this can be changed on the configuration. Therefore, this port needs to be open.
  
  - **A reverse proxy or dedicated subdomain (optional)**  
    As an alternative to opening a non-default port, a reverse proxy can be set up to direct access to SSE endpoint to the Node.js server. Same goes with dedicated subdomain which can be set to handle requests to the SSE endpoint.

### Setup
When technical requirements are already met, to run the SSE endpoint can be done with this example below:

```sh
$ cd WP_PATH/wp-content/plugins/peepso/sse/
$ npm start # using default setting
$ npm start -- --config="../../../peepso/config.js" # alternatively, CLI argument can be set to point to a config file
```

You will see this message below if the SSE endpoint is run correctly:
`PeepSo SSE endpoint is running at http://localhost:${PORT}`

You can then set up a reverse proxy (and maybe a dedicated domain/subdomain) to map request to said domain to the SSE endpoint so that it can be accessed from the internet. If your main domain is `example.com`, your SSE endpoint can be set to `sse.example.com` or anything else you see fit.

After all of the above setup is done, as the last step, you need to point the PeepSo SSE configuration to the SSE endpoint you just created. See the screenshot below for illustration:

<img src="data:image/png;base64,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" alt="" />