<?php
/**
 * Plugin Name: PeepSo Monetization: EDD
 * Plugin URI: https://PeepSo.com
 * Description: <strong>Requires the Easy Digital Downloads plugin.</strong> EDD integration for Your Community.
 * Author: PeepSo
 * Version: 7.0.1.0
 * Author URI: https://PeepSo.com
 * Copyright: (c) 2017 PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later (PHP), property of PeepSo, Inc. (CSS, JavaScript, images)
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: peepso-edd
 * Domain Path: /language
 *
 * The PHP code of this plugin is Open Source. This PHP software can be redistributed and/or modified under the terms of the 
 * GNU General Public License (version 2 or later), as published by the Free Software Foundation. 
 * See the GNU General Public License or the LICENSE file for more details.  
 *  
 * Unless explicitly stated otherwise inside the respective files, all CSS, JavaScript, and images included in this plugin are proprietary to PeepSo, Inc.  
 * They are protected by copyright and other applicable intellectual property laws. 
 * Unauthorized use, reproduction, or distribution of these assets is STRICTLY PROHIBITED.  
 *  
 * "PeepSo" and "Your Community. Your Way." are registered trademarks of PeepSo, Inc.  
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY. 
*/


class PeepSoEDD
{
    private static $_instance = NULL;

    const PLUGIN_EDD = 1483517;
    const PLUGIN_SLUG = 'edd';

    const PLUGIN_NAME    = 'Monetization: EDD';
    const PLUGIN_VERSION = '7.0.1.0';
    const PLUGIN_RELEASE = ''; //ALPHA1, BETA1, RC1, '' for STABLE
    const MODULE_ID      = 1000;

    const THIRDPARTY_MIN_VERSION = '2.9.22';

    /**
     * EDD compatibility version
     */
    const EDD_PURCHASE_POST_META = 'peepso_edd_purchase';

    const PROFILE_TAB_SLUG = 'edit-cc';

    private static function ready_thirdparty() {

        if(defined('EDD_VERSION')) {
            $edd_version = EDD_VERSION;
        } else {
            // EDD 3 fallback, sometimes their EDD_VERSION constant is not present
            // This needs to be improved once MIN_VERSION goes beyond 3.0
            $edd_version = self::THIRDPARTY_MIN_VERSION;
        }

        if ( class_exists('Easy_Digital_Downloads') && version_compare( $edd_version, self::THIRDPARTY_MIN_VERSION, '>=' ) ) {
            return TRUE;
        }

        return FALSE;
    }

	private static function ready() {
        if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION) && self::ready_thirdparty()) {
            return TRUE;
        }
    }

    public function __construct() {
        /** VERSION INDEPENDENT hooks **/

        // Admin
        if (is_admin()) {
            add_action('admin_init', array(&$this, 'peepso_check'));
            add_filter('peepso_license_config', function($list) {
                $list[] = array(
                    'plugin_slug' => self::PLUGIN_SLUG,
                    'plugin_name' => self::PLUGIN_NAME,
                    'plugin_edd' => self::PLUGIN_EDD,
                    'plugin_version' => self::PLUGIN_VERSION
                );
                return $list;
            });
        }

        // Compatibility
        add_filter('peepso_all_plugins', function($plugins){
            $plugins[plugin_basename(__FILE__)] = get_class($this);
            return $plugins;
        });

        // Translations
        add_action('init', function(){
            $path = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'language' . DIRECTORY_SEPARATOR . 'peepso-edd-' . determine_locale() . '.mo';
            load_textdomain('peepso-edd', $path);
        });

        // Activation
        register_activation_hook(__FILE__, array($this, 'activate'));

        /** VERSION LOCKED hooks **/
        if (self::ready()) {
            if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
                return;
            }

            add_action('peepso_init', array(&$this, 'init'));

            // insert a post after purchase
            // moved here for stripe and other payment methods compatibility
            add_action('wp_insert_post', array(&$this, 'after_checkout'), 10, 3);
            add_action('edd_update_payment_status', array(&$this, 'after_checkout_edd3'), 10, 3);
        }
    }

    /**
     * Retrieve singleton class instance
     * @return peepso-edd instance
     */
    public static function get_instance()
    {
        if (NULL === self::$_instance) {
            self::$_instance = new self();
        }
        return (self::$_instance);
    }


    public function init()
    {
        PeepSo::add_autoload_directory(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR);
        PeepSoTemplate::add_template_directory(plugin_dir_path(__FILE__));

        if (is_admin()) {
            add_action('admin_init', array(&$this, 'peepso_check'));

            // add admin tab
            add_filter('peepso_admin_config_tabs', function($tabs){
                $tabs['edd'] = array(
                    'label' => __('EDD', 'peepso-edd'),
                    'icon' => 'https://cdn.peepso.com/icons/plugins/'.self::PLUGIN_EDD.'.svg',
                    'tab' => 'edd',
                    'description' => __('EDD', 'peepso-edd'),
                    'function' => 'PeepSoConfigSectionEDD',
                    'cat'   => 'monetization',
                );

                return $tabs;
            });

            if (isset($_GET['tab']) && $_GET['tab'] == self::PLUGIN_SLUG) {
                // enqueue required
                add_action('admin_enqueue_scripts', function() {
                    wp_enqueue_script(
                        $handle = 'peepso-edd',
                        $src    = plugin_dir_url(__FILE__) . 'assets/js/admin.js',
                        array('peepso'),
                        self::PLUGIN_VERSION,
                        TRUE
                    );
                });
            }

        } else {
            // profile menu
            add_action('peepso_profile_segment_edd', array(&$this, 'peepso_profile_segment_edd'));

            // about tab
            add_filter('peepso_filter_about_tabs', array($this, 'peepso_profile_about_tabs'));

            // edit cc tab
            add_action( 'peepso_profile_segment_' . self::PROFILE_TAB_SLUG, array($this, 'peepso_profile_edd_tabs_content' ) );

            // stream title
            add_filter('peepso_activity_stream_action', array(&$this, 'activity_stream_action'), 10, 2);

            // stream content
            add_action('peepso_activity_post_attachment', array(&$this, 'attach_edd'), 20, 1);

            // enqueue required
            add_action('wp_enqueue_scripts', function() {
                wp_enqueue_script(
                    $handle = 'peepso-edd',
                    $src    = plugin_dir_url(__FILE__) . 'assets/js/peepso-edd.js',
                    array('peepso'),
                    self::PLUGIN_VERSION,
                    TRUE
                );

                wp_register_style(
                    $handle = 'peepso-edd',
                    $src    = plugin_dir_url(__FILE__) . 'assets/css/peepso-edd.css',
                    $deps   = array(),
                    $ver    = time(),
                    $media  = 'all'
                );
                wp_enqueue_style('peepso-edd');
            });

            if (PeepSo::get_option('peepso_edd_display_purchase',1) && PeepSo::get_option('peepso_edd_display_purchase_override', 1)) {
                add_filter('peepso_profile_preferences', array($this, 'filter_profile_preferences'), 99);
            }

            add_filter('peepso_post_filters', array($this, 'post_filters'), 99, 1);
        }

        // PeepSo navigation
        add_filter('peepso_navigation_profile', array(&$this, 'filter_peepso_navigation_profile'));
    }

    public function license_notice()
    {
        PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG);
    }

    public function license_notice_forced()
    {
        PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG, true);
    }


    /**
     * Check if PeepSo class is present (ie the PeepSo plugin is installed and activated)
     * If there is no PeepSo, immediately disable the plugin and display a warning
     * Run license and new version checks against PeepSo.com
     * @return bool
     */
    public function peepso_check()
    {
        if (!class_exists('PeepSo')) {
            add_action('admin_notices', array(&$this, 'peepso_disabled_notice'));
            unset($_GET['activate']);
            deactivate_plugins(plugin_basename(__FILE__));
            return (FALSE);
        }

        if(!self::ready_thirdparty()) {
            add_action('admin_notices',function() {
                if(method_exists('PeepSo','third_party_warning')) {
                    PeepSo::third_party_warning('Easy Digital Downloads','easy-digital-downloads',FALSE,self::THIRDPARTY_MIN_VERSION, self::PLUGIN_NAME);
                }
            }, 10030);
        }


        // PeepSo.com license check
        if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
            add_action('admin_notices', array(&$this, 'license_notice'));
        }

        if (isset($_GET['page']) && 'peepso_config' == $_GET['page'] && !isset($_GET['tab'])) {
            add_action('admin_notices', array(&$this, 'license_notice_forced'));
        }

        // PeepSo.com new version check
        // since 1.7.6
        if(method_exists('PeepSoLicense', 'check_updates_new')) {
            PeepSoLicense::check_updates_new(self::PLUGIN_EDD, self::PLUGIN_SLUG, self::PLUGIN_VERSION, __FILE__);
        }

        return (TRUE);
    }

    /**
     * Display a message about PeepSo not present
     */
    public function peepso_disabled_notice()
    {
        ?>
        <div class="error peepso">
            <strong>
                <?php
                echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepso-edd'), self::PLUGIN_NAME),
                ' <a href="plugin-install.php?tab=plugin-information&amp;plugin=peepso-core&amp;TB_iframe=true&amp;width=772&amp;height=291" class="thickbox">',
                __('Get it now!', 'peepso-edd'),
                '</a>';
                ?>
            </strong>
        </div>
        <?php
    }

    /**
     * Activation hook for the plugin.
     *
     * @since 1.0.0
     */
    public function activate() {

        if (!$this->peepso_check()) {
            return (FALSE);
        }

        require_once(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'install' . DIRECTORY_SEPARATOR . 'activate.php');
        $install = new PeepSoEDDInstall();
        $res = $install->plugin_activation();
        if (FALSE === $res) {
            // error during installation - disable
            deactivate_plugins(plugin_basename(__FILE__));
        }
        return (TRUE);
    }

    /**
     * Profile Segments - add link
     * @param $links
     * @return mixed
     */
    public function filter_peepso_navigation_profile($links) {
        $edd_tabs = $this->get_available_profile_tabs();

        if (count($edd_tabs) > 0 && PeepSoUrlSegments::get_view_id(PeepSoProfileShortcode::get_instance()->get_view_user_id()) === get_current_user_id()) {
            $links['edd'] = array(
                'label'=> __('Purchases', 'peepso-edd'),
                'href' => 'edd',
                'icon' => 'gcis gci-shopping-cart'
            );
        }

        return $links;
    }

    /**
     * Profile Segments - add tab
     * @return mixed
     */
    public function peepso_profile_segment_edd()
    {
        $edd_tabs = $this->get_available_profile_tabs();

        if (count($edd_tabs) > 0) {
            $PeepSoUrlSegments = PeepSoUrlSegments::get_instance();

            reset($edd_tabs);
            $default = key($edd_tabs);
            //$default = array_key_first($edd_tabs);

            $current_tab = $PeepSoUrlSegments->get(3) != '' ? $PeepSoUrlSegments->get(3) : $default;
            $template_tab = 'tab-' . $current_tab;

            $current_url = (isset($_SERVER['HTTPS']) ? "https" : "http") . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
            if (PeepSo::get_option('disable_questionmark_urls', 0) === 1) {
                $current_url .= '?';
            } else {
                $current_url .= '&';
            }

            echo PeepSoTemplate::exec_template('profile', 'profile-edd', array(
                    'tabs' => $edd_tabs,
                    'current_tab' => $current_tab,
                    'template_tab' => $template_tab,
                    'current_url' => $current_url)
            );
        }
    }

    /**
     * Create EDD credit card info tabs content in PeepSo About
     * @param  array $tabs   The profile tabs
     * @return array
     */
    public function peepso_profile_about_tabs($tabs) {
        $PeepSoUser = PeepSoUser::get_instance(get_current_user_id());

        $tabs[self::PROFILE_TAB_SLUG] = [
            'link' => $PeepSoUser->get_profileurl() . 'about/'. self::PROFILE_TAB_SLUG .'/',
            'label' => 'Billing Details',
            'icon' => 'gcis gci-credit-card'
        ];

        // handle submitted data
        $this->save_billing_details();
        return $tabs;
    }

    /**
     * Save default billing details
     */
    public function save_billing_details() {
        $check_slug = PeepSoUrlSegments::get_instance()->get(3);
        if (isset($_POST) && count($_POST) > 0 && $check_slug == self::PROFILE_TAB_SLUG) {
            $input = new PeepSoInput();

            $data = [
                'line1' => $input->value('card_address', '', FALSE),
                'line2' => $input->value('card_address_2', '', FALSE),
                'city' => $input->value('card_city', '', FALSE),
                'zip' => $input->value('card_zip', '', FALSE),
                'country' => $input->value('billing_country', '', FALSE),
                'state' => $input->value('card_state', '', FALSE),
            ];

            update_user_meta(get_current_user_id(), '_edd_user_address', $data);
        }
    }

    /**
     * Change the activity stream item action string
     * @param  string $action The default action string
     * @param  object $post   The activity post object
     * @return string
     */
    public function activity_stream_action($action, $post)
    {
        $edd_payment_post = get_post_meta($post->ID, self::EDD_PURCHASE_POST_META, true);

        if (!empty($edd_payment_post)) {
            return PeepSo::get_option('peepso_edd_display_purchase_content', __('made a purchase'));
        }

        return $action;
    }

    /**
     * Insert PeepSo activity
     * @param   integer $post_id    The post id
     * @param   object  $post       The activity post object
     * @param   boolean $update     Is update or not
     */
    public function after_checkout($post_id, $post, $update)
    {
        if (PeepSo::get_option('peepso_edd_display_purchase', 1) === 1 && $post->post_author !== 0 && $post->post_type == 'edd_payment' && $update) {
            $payment_total = get_post_meta($post_id, '_edd_payment_total', TRUE);

            $this->create_activity($post->post_author, $post_id, $payment_total);
        }
    }

    public function after_checkout_edd3($order_id, $new_status, $old_status)
    {
        if (!self::ready() || !PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG) || !function_exists('edd_get_order')) { return; }

        $order = edd_get_order($order_id);

        if (PeepSo::get_option('peepso_edd_display_purchase', 1) === 1 && $order->user_id !== 0 && $new_status == 'complete') {
            $this->create_activity($order->user_id, $order_id, $order->total);
        }
    }

    public function create_activity($user_id, $post_id, $payment_total)
    {
        // if override enabled
        $user_meta = get_user_meta($user_id, 'peepso_edd_post_to_stream', TRUE);

        if (empty($user_meta) && $user_meta !== '' && PeepSo::get_option('peepso_edd_display_purchase_override', 1) === 1) {
            return;
        } else {
            // if free purchase?
            if ($payment_total === '' || (intval($payment_total) === 0 && PeepSo::get_option('peepso_edd_display_free_purchase', 1) === 0)) {
                return;
            } else {
                add_filter('peepso_activity_allow_empty_content', array(&$this, 'activity_allow_empty_content'), 10, 1);

                $peepso_activity    = new PeepSoActivity();
                $peepso_activity_id = $peepso_activity->add_post($user_id, $user_id, '', [
                    'module_id' => self::MODULE_ID,
                ]);
                add_post_meta($peepso_activity_id, self::EDD_PURCHASE_POST_META, $post_id);

                remove_filter('peepso_activity_allow_empty_content', array(&$this, 'activity_allow_empty_content'));
            }
        }
    }

    /**
     * Filter to allow empty content
     * @param   boolean $allowed     Is allowed
     * @return  boolean
     */
    public function activity_allow_empty_content($allowed)
    {
        return TRUE;
    }

    /**
     * Filter to attach purchased items
     * @param   object  $post       The activity post object
     * @return  mixed
     */
    public function attach_edd($post)
    {
        $edd_payment_post = get_post_meta($post->ID, self::EDD_PURCHASE_POST_META, TRUE);

        if ($edd_payment_post) {
            // get purchased items
            $post_meta = get_post_meta($edd_payment_post, '_edd_payment_meta', TRUE);
            if (isset($post_meta['downloads']) && count($post_meta['downloads']) > 0) {
                $item_content = '';

                foreach ($post_meta['downloads'] as $download) {
                    $item_permalink = get_permalink($download['id']);

                    $item_content .= '<div class="ps-edd__slider-item">';
                    $item_content .= ps_oembed_get($item_permalink, array('width' => 500, 'height' => 300));
                    $item_content .= '</div>';
                }

                $content = '<div class="ps-edd-product">';
                $content .= '<div class="ps-edd__slider-wrapper" aria-label="' . __('Purchase slider', 'peepso-edd') . '">';
                $content .= '<div class="ps-edd__slider">' . $item_content . '</div>';
                if (count($post_meta['downloads']) > 1) {
                    $content .= '<button class="ps-edd__slider-btn ps-edd__slider-btn--prev" data-step="-1" aria-label="' . __('Prev', 'peepso-edd') . '"><i class="ps-icon-caret-left"></i></button>';
                    $content .= '<button class="ps-edd__slider-btn ps-edd__slider-btn--next" data-step="1" aria-label="' . __('Next', 'peepso-edd') . '"><i class="ps-icon-caret-right"></i></button>';
                }
                $content .= '</div>';
                $content .= '</div>';

                echo $content;
            }
        }
    }

    /**
     * Find available tabs for profile
     * @return  array
     */
    private function get_available_profile_tabs()
    {
        $PeepSoUser = PeepSoUser::get_instance(PeepSoProfileShortcode::get_instance()->get_view_user_id());
        $user_url = $PeepSoUser->get_profileurl();

        $tabs = PeepSo::get_option('peepso_edd_profile_tabs', array());
        $tabs = explode(PHP_EOL, $tabs);

        $edd_tabs = array();
        foreach ($tabs as $tab) {
            $tab = trim($tab);
            switch ($tab) {
                case 'downloads':
                    $label = PeepSo::get_option('peepso_edd_tab_downloads_label', '');
                    $edd_tabs['downloads'] = [
                        'link' => $user_url . 'edd/downloads/',
                        'label' => !empty($label) ? $label : __('Downloads', 'peepso-edd'),
                    ];
                    break;
                case 'orders':
                    $label = PeepSo::get_option('peepso_edd_tab_orders_label', '');
                    $edd_tabs['orders'] = [
                        'link' => $user_url . 'edd/orders/',
                        'label' => !empty($label) ? $label : __('Orders', 'peepso-edd'),
                    ];
                    break;
                case 'licenses':
                    if (class_exists('EDD_Software_Licensing')) {
                        $label = PeepSo::get_option('peepso_edd_tab_licenses_label', '');
                        $edd_tabs['licenses'] = [
                            'link' => $user_url . 'edd/licenses/',
                            'label' => !empty($label) ? $label : __('Licenses', 'peepso-edd'),
                        ];
                    }
                    break;
                case 'subscriptions':
                    if (class_exists('EDD_Recurring')) {
                        $label = PeepSo::get_option('peepso_edd_tab_subscriptions_label', '');
                        $edd_tabs['subscriptions'] = [
                            'link' => $user_url . 'edd/subscriptions/',
                            'label' => !empty($label) ? $label : __('Subscriptions', 'peepso-edd'),
                        ];
                    }
                    break;
                case 'all-access':
                    if (class_exists('EDD_All_Access')) {
                        $label = PeepSo::get_option('peepso_edd_tab_all_access_label', '');
                        $edd_tabs['all-access'] = [
                            'link' => $user_url . 'edd/all-access/',
                            'label' => !empty($label) ? $label : __('All Access Downloads', 'peepso-edd'),
                        ];
                    }
                    break;
            }
        }

        return $edd_tabs;
    }

    /**
     * Return preferences for profile
     * @return  array
     */
    public function filter_profile_preferences($pref)
    {
        $user_meta = get_user_meta(get_current_user_id(), 'peepso_edd_post_to_stream', TRUE);
        $pref['edd_activity_settings'] = array(
            'title' => __('Shopping Activity', 'peepso-edd'),
            'items' => array(
                'peepso_edd_post_to_stream' => array(
                    'label-desc' => __('Share my new purchases to my feed', 'peepso-edd'),
                    'type'       => 'yesno_switch',
                    'value'      => $user_meta !== '' ? intval($user_meta) : 1,
                    'loading'    => true,
                )
            )
        );
        return $pref;
    }

    public function post_filters($options) {
        if (self::MODULE_ID == intval($options['post']->act_module_id) || get_post_meta($options['post']->ID, self::EDD_PURCHASE_POST_META)) {
            if (isset($options['acts']['edit'])) {
                unset($options['acts']['edit']);
            }
        }
    
        return $options;
    }

}

PeepSoEDD::get_instance();
