<?php
/**
 * Plugin Name: PeepSo Monetization: WPAdverts
 * Plugin URI: https://PeepSo.com
 * Description: Classifieds integration for your PeepSo community. Requires the WpAdverts plugin.
 * Tags: peepso, wpadverts, integration
 * Author: PeepSo
 * Version: 7.0.1.0
 * Author URI: https://PeepSo.com
 * Copyright: (c) PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later (PHP), property of PeepSo, Inc. (CSS, JavaScript, images)
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: peepso-wpadverts
 * Domain Path: /language
 *
 * The PHP code of this plugin is Open Source. This PHP software can be redistributed and/or modified under the terms of the 
 * GNU General Public License (version 2 or later), as published by the Free Software Foundation. 
 * See the GNU General Public License or the LICENSE file for more details.  
 *  
 * Unless explicitly stated otherwise inside the respective files, all CSS, JavaScript, and images included in this plugin are proprietary to PeepSo, Inc.  
 * They are protected by copyright and other applicable intellectual property laws. 
 * Unauthorized use, reproduction, or distribution of these assets is STRICTLY PROHIBITED.  
 *  
 * "PeepSo" and "Your Community. Your Way." are registered trademarks of PeepSo, Inc.  
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY. 
*/

class PeepSoWPAdverts {

    private static $_instance = NULL;

    const PLUGIN_EDD = '607967';
    const PLUGIN_SLUG = 'PeepSo-WPAdverts';

    const PLUGIN_NAME    = 'Monetization: WPAdverts';
    const PLUGIN_VERSION = '7.0.1.0';
    const PLUGIN_RELEASE = ''; //ALPHA1, BETA10, RC1, '' for STABLE
    const MODULE_ID      = 1002;

    const PLUGIN_DIR_PATH = '';

    // post meta key for photo type (avatar/cover)
    const POST_META_KEY_WPADVERTS_TYPE          = '_peepso_wpadverts_type';
    const POST_META_KEY_WPADVERTS_TYPE_CLASSIFIEDS   = '_peepso_wpadverts_type_classifieds';
    const POST_META_KEY_WPADVERTS_CLASSIFIEDS_ID     = '_peepso_wpadverts_classifieds_id';

    public $shortcodes= array(
        'peepso_wpadverts' => 'PeepSoWPAdvertsShortcode::shortcode_wpadverts',
    );

    public $view_user_id;
    public $classifieds_id;

    private static function ready_thirdparty() {
        $result = TRUE;

        if (!function_exists('adverts_init')) {
            $result = FALSE;
        }

        return $result;
    }

    const THIRDPARTY_MIN_VERSION = '1.4.0';

	private static function ready() {
        if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION) && self::ready_thirdparty()) {
            return TRUE;
        }
    }

    private function __construct() {

        /** VERSION INDEPENDENT hooks **/

        // Admin
        add_filter('peepso_license_config', function($list){
            $list[] = array(
                'plugin_slug' => self::PLUGIN_SLUG,
                'plugin_name' => self::PLUGIN_NAME,
                'plugin_edd' => self::PLUGIN_EDD,
                'plugin_version' => self::PLUGIN_VERSION
            );
            return ($list);
        }, 160);

        if (is_admin()) {
            add_action('admin_init', array(&$this, 'peepso_check'));
        }

        // Compatibility
        add_filter('peepso_all_plugins', function($plugins) {
            $plugins[plugin_basename(__FILE__)] = get_class($this);
            return $plugins;
        });

        // Translations
        add_action('init', array(&$this, 'load_textdomain'));

        // Activation
        register_activation_hook( __FILE__, array( $this, 'activate' ) );

        /** VERSION LOCKED hooks **/
        if(self::ready()) {
            // PeepSo.com license check
			if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
				return;
			}

            if (is_admin()) {
                add_action('peepso_config_before_save-wpadverts', array(&$this, 'before_save_wpadverts'));
            }

            add_action('peepso_init', array(&$this, 'init'));

            // Hook into PeepSo routing, enables single item view (eg /wpadverts/?1235/)
            add_filter('peepso_check_query', array(&$this, 'filter_check_query'), 10, 3);


            add_filter('peepso_activity_remove_shortcode', array(&$this, 'peepso_activity_remove_shortcode'));
            add_filter('peepso_filter_shortcodes', function($list) { return array_merge($list, $this->shortcodes);});
        }
    }

    /**
     * Retrieve singleton class instance
     * @return PeepSoWPAdverts instance
     */
    public static function get_instance()
    {
        if (NULL === self::$_instance) {
            self::$_instance = new self();
        }
        return (self::$_instance);
    }

    public function init()
    {
        PeepSo::add_autoload_directory(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR);
        PeepSoTemplate::add_template_directory(plugin_dir_path(__FILE__));

        if (PeepSo::get_option('wpadverts_user_classifieds_enable', 1)) {
            // PeepSo Navigation
            add_filter('peepso_navigation', array(&$this, 'filter_peepso_navigation'));
            add_filter('peepso_navigation_profile', array(&$this, 'filter_peepso_navigation_profile'));
        }

        if (is_admin()) {
            add_action('admin_init', array(&$this, 'peepso_check'));

            // config tabs
            add_filter('peepso_admin_config_tabs', array(&$this, 'admin_config_tabs'), -1);

            // If post to stream option disabled, don't publish it.
            if ( PeepSo::get_option('wpadverts_post_to_stream_enable', 1) ) {
                // Category settings UI
                add_action ( 'advert_category_edit_form_fields', function($term) {
                    ?>
                    <tr class="form-field">
                        <th scope="row" valign="top"><label for="cat_Image_url"><?php echo __('PeepSo WP Adverts integration'); ?></label></th>
                        <td>
                            <select name="peepso_wpadverts" id="peepso_wpadverts" style="width:60%;">
                                <option value="1"><?php echo __('Enabled','peepso-wpadverts');?></option>
                                <option value="0" <?php if(!PeepSoWPAdverts::enabled_for_category($term->term_id)) { echo "selected"; }?> ><?php echo __('Disabled','peepso-wpadverts');?></option>
                            </select>

                            <br />
                            <span class="description"><?php echo __('Use with caution! Disabling PeepSo WPAdverts integration for a category will prevent new stream posts from being created.','peepso-wpadverts'); ?></span>
                        </td>
                    </tr>
                    <?php
                }, 20, 1);

                // Category settings save
                add_action ( 'edited_advert_category', function( $term_id ) {
                    if ( isset( $_POST['peepso_wpadverts'] ) ) {
                        //save the option array
                        update_option( "peepso_wpadverts_category_$term_id", $_POST['peepso_wpadverts'] );
                    }
                });
            }
        } else {

            // post to activity
            add_action('peepso_activity_post_attachment', array(&$this, 'attach_classifieds'), 20, 1);
            add_filter('peepso_activity_stream_action', array(&$this, 'activity_stream_action'), 10, 2);
            add_filter('peepso_activity_post_actions',      array(&$this, 'modify_post_actions'),50); // priority set to last

            // If User Classifieds Enable
            if (PeepSo::get_option('wpadverts_user_classifieds_enable', 1)) {
                // Profile Slug
                $profile_slug = PeepSo::get_option('wpadverts_navigation_profile_slug', 'classifieds', TRUE);
                add_action('peepso_profile_segment_'.$profile_slug,     array(&$this, 'action_profile_segment_classifieds'));

                // Register Shortcodes
                PeepSoWPAdvertsShortcode::register_shortcodes();
            }

            // overrides wpadverts template
            if (!PeepSo::get_option('wpadverts_overrides_disable', 0)) {
                add_action("adverts_template_load", array(&$this, "custom_list_template"));
            }
            add_action('wp_enqueue_scripts', array(&$this, 'enqueue_scripts'));

            if(1 == PeepSo::get_option('wpadverts_chat_enable', 0) ) {
                add_action('adverts_tpl_single_bottom', function() {
                    global $post;
                    if('advert' != $post->post_type || !get_current_user_id() || $post->post_author == get_current_user_id() || !class_exists('PeepSoMessages')) { return; }
                    ?>
                    <a href="#" class="ps-js-wpadverts-message" data-id="<?php echo $post->post_author;?>" data-post-id="<?php echo $post->ID ?>">
                        <i class="ps-icon-envelope-alt"></i><span><?php echo __('Send Message', 'peepso-wpadverts');?></span>
                        <img src="<?php echo PeepSo::get_asset('images/ajax-loader.gif'); ?>" style="display:none" /></a>
                    <?php
                    self::get_instance()->enqueue_scripts();
                    wp_enqueue_script('peepso-wpdverts-classifieds');
                    echo '<div style="display:none">';
                    do_action('peepso_activity_dialogs');
                    echo '</div>';
                }, 99);
            }

            if (PeepSo::get_option('disable_questionmark_urls', 0) === 1 && !wp_doing_ajax() && strpos($_SERVER['REQUEST_URI'], 'ajax' ) === FALSE) {
                add_action('dynamic_sidebar_before', function($index) {
                    global $wp_registered_widgets;

                    $sidebars_widgets = wp_get_sidebars_widgets();
                    $widget_exist = FALSE;

                    foreach ((array) $sidebars_widgets[$index] as $id) {
                        if (!isset($wp_registered_widgets[$id] ) ) {
                            continue;
                        }

                        if (!$widget_exist && isset($wp_registered_widgets[$id]['id']) && strpos($wp_registered_widgets[$id]['id'], 'wpadverts-widget-ads') !== FALSE) {
                            $widget_exist = TRUE;
                        }
                    }

                    if ($widget_exist) {
                        // save original wp_query
                        if (!isset($GLOBALS['_wpadverts_sidebar_index'])) {
                            $GLOBALS['_wpadverts_sidebar_index'] = [];
                        }
                        $GLOBALS['_wpadverts_sidebar_index'][] = $index;
                        $GLOBALS['_wp_query'] = $GLOBALS['wp_query'];
                        $GLOBALS['_post'] = $GLOBALS['post'];
                    }
                });

                add_action('dynamic_sidebar_after', function($index) {
                    if (isset($GLOBALS['_wpadverts_sidebar_index']) && in_array($index, $GLOBALS['_wpadverts_sidebar_index'])) {
                        // restore wp_query
                        $GLOBALS['wp_query'] = $GLOBALS['_wp_query'];
                        $GLOBALS['post'] = $GLOBALS['_post'];
                    }
                });
            }

            add_filter('peepso_post_filters', array($this, 'post_filters'), 99, 1);
            add_action('peepso_before_get_posts', function() {
                // This action is required to remove post status other than "publish" and "future"
                global $wp_post_statuses;

                $allowed_status = ['publish', 'future'];
                if ($allowed_status) {
                    $new_status = [];
                    foreach ($allowed_status as $status) {
                        if (isset($wp_post_statuses[$status])) {
                            $new_status[$status] = $wp_post_statuses[$status];
                        }
                    }

                    $GLOBALS['wp_post_statuses'] = $new_status;
                }
            });
        }

        add_filter('wp_ajax_peepso_wpadverts_ad_info', function () {
            $id = (int) $_POST['id'];
            $ad_post = get_post($id);
            $result = null;

            if ($ad_post instanceof WP_Post) {
                $result = [
                    'title' => wp_trim_words($ad_post->post_title, 4),
                    'description' => wp_trim_words($ad_post->post_content, 10),
                    'html_chat' => PeepSoTemplate::exec_template('wpadverts', 'chat-attachment', array('post' => $ad_post, 'btn_remove' => TRUE), TRUE),
                    'html_message' => PeepSoTemplate::exec_template('wpadverts', 'message-attachment', array('post' => $ad_post, 'btn_remove' => TRUE), TRUE),
                ];
            }

            die(json_encode($result));
        });

        add_action('peepso_activity_after_add_post', function($post_id) {
            $input = new PeepSoInput();
            $id = (int) $input->int('wpadverts', 0);
            $ad_post = get_post($id);
            if ($ad_post instanceof WP_Post) {
                delete_post_meta($post_id, self::POST_META_KEY_WPADVERTS_TYPE . '_attachment');
                add_post_meta($post_id, self::POST_META_KEY_WPADVERTS_TYPE . '_attachment', $id, TRUE);
            }
        }, 20);

        add_action('peepso_message_attachment_before', function($post_id) {
            $ad_post_id = (int) get_post_meta($post_id, self::POST_META_KEY_WPADVERTS_TYPE . '_attachment', TRUE);
            if ($ad_post_id) {
                $ad_post = get_post($ad_post_id);
                if ($ad_post instanceof WP_Post) {
                    echo PeepSoTemplate::exec_template('wpadverts', 'chat-attachment', array('post' => $ad_post), TRUE);
                }
            }
        });

        add_action( 'save_post', array(&$this, 'save_classifieds'), 10, 3 );

        // delete stream when ad deleted
        add_action('before_delete_post', array(&$this, 'delete_stream_classifieds'), 10, 1);
    }

    // Check if integration is enabled for the post, based on its categories
    public static function enabled_for_post_categories($post_id) {
        $cats = get_the_terms($post_id, 'advert_category');
        if ( $cats && ! is_wp_error( $cats ) ) {
            foreach($cats as $cat) {
                if(!PeepSoWPAdverts::enabled_for_category($cat->term_id))
                {
                    return FALSE;
                }
            }
        }

        return TRUE;
    }

    // Checks if integration is enabled for given category ID
    public static function enabled_for_category($id) {
        return get_option( "peepso_wpadverts_category_$id", 1);
    }

    public function peepso_activity_remove_shortcode( $content )
    {
        foreach($this->shortcodes as $shortcode=>$class) {
            foreach($this->shortcodes as $shortcode=>$class) {
                $from = array('['.$shortcode.']','['.$shortcode);
                $to = array('&#91;'.$shortcode.'&#93;', '&#91;'.$shortcode);
                $content = str_ireplace($from, $to, $content);
            }
        }
        return $content;
    }

    public function before_save_wpadverts() {
        $_POST['wpadverts_navigation_profile_slug'] = strtolower(preg_replace('/[^a-z0-9\-\_\.]+/i', "", $_POST['wpadverts_navigation_profile_slug']));
    }

    public function license_notice()
    {
        PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG);
    }

    public function license_notice_forced()
    {
        PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG, true);
    }

    public function filter_check_query($sc, $page, $url)
    {
        if(PeepSoWPAdvertsShortcode::SHORTCODE == $page ) {
            $sc = PeepSoWPAdvertsShortcode::get_instance();
            $sc->set_page($url);
        }
    }

    /**
     * Loads the translation file for the PeepSo plugin
     */
    public function load_textdomain()
    {
        $path = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'language' . DIRECTORY_SEPARATOR . 'peepso-wpadverts-' . determine_locale() . '.mo';
        load_textdomain('peepso-wpadverts', $path);
    }

    /**
     * Hooks into PeepSo for compatibility checks
     * @param $plugins
     * @return mixed
     */
    public function filter_all_plugins($plugins)
    {
        $plugins[plugin_basename(__FILE__)] = get_class($this);
        return $plugins;
    }

    /**
     * Check if PeepSo class is present (ie the PeepSo plugin is installed and activated)
     * If there is no PeepSo, immediately disable the plugin and display a warning
     * Run license and new version checks against PeepSo.com
     * @return bool
     */
    public function peepso_check()
    {
        if (!class_exists('PeepSo')) {
            add_action('admin_notices', array(&$this, 'peepso_disabled_notice'));
            unset($_GET['activate']);
            deactivate_plugins(plugin_basename(__FILE__));
            return (FALSE);
        }

        if (!self::ready_thirdparty()) {

            add_action('admin_notices', function() {
                if(method_exists('PeepSo','third_party_warning')) {
                    PeepSo::third_party_warning('WP Adverts','wpadverts',FALSE,self::THIRDPARTY_MIN_VERSION, self::PLUGIN_NAME);
                }
            }, 10100);
        }

        // PeepSo.com license check
        if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
             add_action('admin_notices', array(&$this, 'license_notice'));
        }

        if (isset($_GET['page']) && 'peepso_config' == $_GET['page'] && !isset($_GET['tab'])) {
            add_action('admin_notices', array(&$this, 'license_notice_forced'));
        }

        // PeepSo.com new version check
        // since 1.7.6
        if(method_exists('PeepSoLicense', 'check_updates_new')) {
            PeepSoLicense::check_updates_new(self::PLUGIN_EDD, self::PLUGIN_SLUG, self::PLUGIN_VERSION, __FILE__);
        }

        return (TRUE);
    }

    /**
     * Display a message about PeepSo not present
     */
    public function peepso_disabled_notice()
    {
        ?>
        <div class="error peepso">
            <strong>
                <?php
				echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepso-wpadverts'), self::PLUGIN_NAME),
                    ' <a href="plugin-install.php?tab=plugin-information&amp;plugin=peepso-core&amp;TB_iframe=true&amp;width=772&amp;height=291" class="thickbox">',
                    __('Get it now!', 'peepso-wpadverts'),
                    '</a>';
                ?>
                <?php //echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepso-wpadverts'), self::PLUGIN_NAME);?>
            </strong>
        </div>
        <?php
    }


    /**
     * Activation hook for the plugin.
     *
     * @since 1.0.0
     */
    public function activate() {

        if (!$this->peepso_check()) {
            return (FALSE);
        }

        require_once(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'install' . DIRECTORY_SEPARATOR . 'activate.php');
        $install = new WPAdvertsPeepSoInstall();
        $res = $install->plugin_activation();
        if (FALSE === $res) {
            // error during installation - disable
            deactivate_plugins(plugin_basename(__FILE__));
        }
        return (TRUE);
    }

    public function admin_enqueue_scripts()
    {
    	//
    }

    /**
     * Enqueue custom scripts and styles
     *
     * @since 1.0.0
     */
    public function enqueue_scripts()
    {
        wp_enqueue_script('peepso-wpdverts-classifieds',
            PeepSo::get_asset('js/bundle.min.js', __FILE__),
            array('peepso', 'peepso-page-autoload'), self::PLUGIN_VERSION, TRUE);

        wp_localize_script('peepso', 'peepsowpadvertsdata', array(
            'listItemTemplate' => PeepSoTemplate::exec_template('wpadverts', 'wpadverts-item', NULL, TRUE),
            'user_id' => get_current_user_id(),
            'module_id' => self::MODULE_ID,
            'lang' => array(
                'more' => __('More', 'peepso-wpadverts'),
                'less' => __('Less', 'peepso-wpadverts'),
                'member' => __('member', 'peepso-wpadverts'),
                'members' => __('members', 'peepso-wpadverts'),
            )
        ));
    }

    /**
     * Publish post
     *
     * @param int $post_id The post ID.
     * @param post $post The post object.
     * @param bool $update Whether this is an existing post being updated or not.
     */
    public function save_classifieds($post_id, $post, $update)
    {
        /*
         * In production code, $slug should be set only once in the plugin,
         * preferably as a class property, rather than in each function that needs it.
         */
        $post_type = get_post_type($post_id);

        // If this isn't a 'advert' post, don't publish it.
        if ( "advert" != $post_type ) return;

        // If post to stream option disabled, don't publish it.
        if ( !PeepSo::get_option('wpadverts_post_to_stream_enable', 1) ) return;

        // If moderation enabled and post_status is `advert-pending`
        if (PeepSo::get_option('wpadverts_moderation_enable', 1) && $post->post_status == 'advert-pending') return;

        // #5579 exclude selected categories
        if(!PeepSoWPAdverts::enabled_for_post_categories($post_id)) { return; }

        // If post status is pending
        if ( strpos($post->post_status, 'pending') !== FALSE || $post->post_status == 'advert_tmp' ) return;

        // If stream already posted. skip creating new stream.
        $args = array(
            'post_type' => PeepSoActivityStream::CPT_POST,
            'meta_query' => array(
                array(
                    'key'     => self::POST_META_KEY_WPADVERTS_CLASSIFIEDS_ID,
                    'value'   => $post_id,
                    'compare' => '=',
                ),
            ),
        );

        $existing_post = get_posts( $args );
        if( $existing_post ) {
            if($post->post_status == 'trash') {
                $status = 'trash';
            } elseif($post->post_status == 'expired') {
                $status = 'expired';
            }else {
                $status = 'publish';
            }

            // #5579 exclude selected categories
            if(!PeepSoWPAdverts::enabled_for_post_categories($existing_post[0]->ID)) {
                // force unpublish if activity already created and category integration is disabled
                $status = 'pending';
            }

            // Update the post into the database
            $ad_post = array(
                'ID'           => $existing_post[0]->ID,
                'post_status'  => $status,
                'post_date'		=> $post->post_date,
                'post_date_gmt' => $post->post_date_gmt,
            );
            wp_update_post( $ad_post );
            return ;
        }

        // - Post to stream
        $this->classifieds_id = $post_id;
        add_filter('peepso_activity_allow_empty_content', array(&$this, 'activity_allow_empty_content'), 10, 1);

        $user_id = $post->post_author;
        $content = '';
        $extra = array(
            'module_id'     => self::MODULE_ID,
            'act_access'    => PeepSo::get_option('wpadverts_post_privacy', PeepSo::ACCESS_PUBLIC),
            'post_date'     => $post->post_date,
            'post_date_gmt' => $post->post_date_gmt,
            'post_status'   => $post->post_status
        );

        $peepso_activity = PeepSoActivity::get_instance();
        $stream_id = $peepso_activity->add_post($user_id, $user_id, $content, $extra);
        add_post_meta($stream_id, self::POST_META_KEY_WPADVERTS_TYPE, self::POST_META_KEY_WPADVERTS_TYPE_CLASSIFIEDS, true);
        add_post_meta($stream_id, self::POST_META_KEY_WPADVERTS_CLASSIFIEDS_ID, $post_id);

        remove_filter('peepso_activity_allow_empty_content', array(&$this, 'activity_allow_empty_content'));

    }

    /**
     * Checks if empty content is allowed
     * @param string $allowed
     * @return boolean always returns TRUE
     */
    public function activity_allow_empty_content($allowed)
    {
        if(isset($this->classifieds_id)) {
            $allowed = TRUE;
        }

        return ($allowed);
    }

    /**
     * Delete stream
     * @param int $post_id
     */
    public function delete_stream_classifieds($post_id)
    {
        /*
         * In production code, $slug should be set only once in the plugin,
         * preferably as a class property, rather than in each function that needs it.
         */
        $post_type = get_post_type($post_id);
        $post = get_post($post_id);

        // If this isn't a 'advert' post, don't publish it.
        if ( "advert" != $post_type ) return;

        // If stream exists delete activity.
        $args = array(
            'post_status'      => array('publish', 'pending', 'trash'),
            'post_type'        => PeepSoActivityStream::CPT_POST,
            'meta_query' => array(
                array(
                    'key'     => self::POST_META_KEY_WPADVERTS_CLASSIFIEDS_ID,
                    'value'   => $post_id,
                    'compare' => '=',
                ),
            ),
        );

        $existing_post = get_posts( $args );
        if( !$existing_post ) return ;

        global $wpdb;

        $act = $wpdb->get_row( $wpdb->prepare( "SELECT * from $wpdb->prefix" . PeepSoActivity::TABLE_NAME . " WHERE act_external_id= %s AND act_module_id = %s AND act_owner_id = %s", $existing_post[0]->ID, self::MODULE_ID, $existing_post[0]->post_author));
        if( !$act ) return ;
        $activity = PeepSoActivity::get_instance();
        $activity->delete_activity($act->act_id);

        // In some setups, this line causes posts being randomly deleted. This constant patches it up.
        if(!defined('PEEPSO_WPADS_DELETE_OVERRIDE') || !PEEPSO_WPADS_DELETE_OVERRIDE) {
            $activity->delete_post($existing_post[0]->ID);
        }
    }

    /**
     * BACKEND SETTINGS
     * ================
     */

    /**
     * Registers a tab in the PeepSo Config Toolbar
     * PS_FILTER
     *
     * @param $tabs array
     * @return array
     */
    public function admin_config_tabs( $tabs )
    {
        $tabs['wpadverts'] = array(
            'label' => __('WPAdverts', 'peepso-wpadverts'),
            'icon' => 'https://cdn.peepso.com/icons/plugins/'.self::PLUGIN_EDD.'.svg',
            'tab' => 'wpadverts',
            'description' => __('PeepSo - WPAdverts Integration', 'peepso-wpadverts'),
            'function' => 'PeepSoConfigSectionWPAdverts',
            'cat'   => 'monetization',
        );

        return $tabs;
    }

    /**
     * FRONTEND
     * ========
     *
     */


    /**
     * Attach the ads to the post display
     * @param  object $post The post
     */
    public function attach_classifieds($stream_post = NULL)
    {
        $ads_type = get_post_meta($stream_post->ID, self::POST_META_KEY_WPADVERTS_TYPE, true);

        if($ads_type === self::POST_META_KEY_WPADVERTS_TYPE_CLASSIFIEDS) {
            $ads_id = get_post_meta($stream_post->ID, self::POST_META_KEY_WPADVERTS_CLASSIFIEDS_ID, true);

            $post = get_post( $ads_id );
            $type = $post->post_type;
            $expires = get_post_meta( $post->ID, "_expiration_date", true );

            $image = get_the_post_thumbnail_url( $ads_id, 'full' );
            $price = adverts_get_the_price( $ads_id );
            $date = date_i18n( get_option( 'date_format' ), get_post_time( 'U', false, $post->ID ) );
            $location = get_post_meta( $post->ID, "adverts_location", true );
            $expires = esc_html( apply_filters( 'adverts_sh_manage_date', date_i18n( get_option('date_format'), $expires ), $post ) );
            $class_featured = '';
            if( $post->menu_order ) {
                $class_featured = ' ps-classified--featured ';
            }

            $is_sold = "";
      			if (class_exists("Wpadverts_Mas")) {
      				$is_sold = get_post_meta( $post->ID, "marked_as_sold", true );
      			}

            $class_sold = "";
            if($is_sold) {
              $class_sold = "ps-classified--sold";
            }

            if ($is_sold) {
              if( adverts_config( "mas.details_show_top_notification")) {
                $message_sold_top = Wpadverts_Mas::get_default_notification();

                if( adverts_config( 'mas.details_top_notification_text' ) ) {
                    $message_sold_top = adverts_config( 'mas.details_top_notification_text' );
                }
              }

              if( adverts_config( "mas.details_show_bottom_notification")) {
                $message_sold_bottom = Wpadverts_Mas::get_default_notification();

                if( adverts_config( 'mas.details_bottom_notification_text' ) ) {
                    $message_sold_bottom = adverts_config( 'mas.details_bottom_notification_text' );
                }
              }
            }

            $target = '';
            $link_target = (int) PeepSo::get_option('site_activity_open_links_in_new_tab', 1);
            if (1 === $link_target) {
                $target = ' target="_blank"';
            }

            // Setup our entry content
            $content = '<div class="ps-classified ps-classified--post ' . $class_featured . $class_sold . '">';
              if ($is_sold) {
                if( adverts_config( "mas.details_show_top_notification")) {
                  $content .= '<div class="ps-classified__label ps-classified__label--sold ps-classified__label--sold-top">';
            			   $content .= $message_sold_top;
                  $content .= '</div>';
                }
              }
              $content .= '<div class="ps-classified__inner">';
                if($image) {
                $content .= '<div class="ps-classified__photo">';
                  $content .= '<a class="ps-classified__photo-link" style="background-image:url(' . get_the_post_thumbnail_url( $ads_id, 'full' ) . ');" href="'. get_permalink( $ads_id ) . '"' . $target . '><img src="' . get_the_post_thumbnail_url( $ads_id, 'full' ) . '" /></a>';
                $content .= '</div>';
                }
                $content .= '<div class="ps-classified__body">';

                    $content .= '<a class="ps-classified__title" href="'. get_permalink( $ads_id ) . '"' . $target . '>'. $post->post_title . '</a>';

                    $content .= '<div class="ps-classified__content">' . wpautop( $post->post_content ) . '</div>';
                    $content .= '<div class="ps-classified__meta-wrapper">';
                        $content .= '<span class="ps-classified__meta ps-classified__meta-date"><i class="gcis gci-clock"></i> '. $date .'</span> <span class="ps-classified__meta ps-classified__meta-location"><i class="gcis gci-map-marker-alt"></i> '. $location .'</span> ';
                        if((get_current_user_id() == $post->post_author) || is_admin()) {
                            $content .= '<span class="ps-classified__meta ps-classified__meta-status"><i class="gcis gci-history"></i> '. __('expires', 'peepso-wpadverts') .': '. $expires .'</span>';
                        }
                    $content .= '</div>';

                    $content .= '<div class="ps-classified__actions">';
                        if($price) {
                            $content .= '<div class="ps-classified__price"><a href="'. get_permalink( $ads_id ) . '"' . $target . '>'.$price.'</a></div>';
                        } else if(adverts_config('empty_price')) {
                            $content .= '<div class="ps-classified__price"><a href="#">' . esc_html(adverts_config('empty_price')) . '</a></div>';
                        }

                        $content .= '<div class="ps-classified__actions-inner ps-btn__group">';
                          if (class_exists('PeepSoMessages') && get_current_user_id() != $post->post_author && 1 == PeepSo::get_option('wpadverts_chat_enable', 0)) {
                              $content .= '<a href="#" class="ps-classified__action ps-btn ps-btn--sm ps-btn--cp ps-btn--app ps-js-wpadverts-message" data-id="' . $post->post_author . '" data-post-id="' . $post->ID . '">';
                              $content .= '<i class="gcis gci-envelope"></i><span>' . __('Send Message', 'peepso-wpadverts') . '</span> ';
                              $content .= '<img src="' . PeepSo::get_asset('images/ajax-loader.gif') . '" style="display:none" /></a> ';
                          }
                          $content .= '<a href="' . get_permalink( $ads_id ) . '"' . $target . ' class="ps-classified__action ps-btn ps-btn--sm ps-btn--cp ps-btn--app"><i class="gcis gci-info-circle"></i><span>' . __('More', 'peepso-wpadverts') . '</span></a>';
                        $content .= '</div>';
                    $content .= '</div>';
                $content .= '</div>';
              $content .= '</div>';

              if ($is_sold) {
                if( adverts_config( "mas.details_show_bottom_notification")) {
                  $content .= '<div class="ps-classified__label ps-classified__label--sold ps-classified__label--sold-bottom">';
            			   $content .= $message_sold_bottom;
                  $content .= '</div>';
                }
              }

            $content .= '</div>';

            echo $content;

            // enqueue script handler
            wp_enqueue_script('peepso-wpdverts-classifieds');
        }
    }

    /**
     * Change the activity stream item action string
     * @param  string $action The default action string
     * @param  object $post   The activity post object
     * @return string
     */
    public function activity_stream_action($action, $post)
    {
        if (self::MODULE_ID === intval($post->act_module_id)) {

            $ads_type = get_post_meta($post->ID, self::POST_META_KEY_WPADVERTS_TYPE, true);
            if($ads_type === self::POST_META_KEY_WPADVERTS_TYPE_CLASSIFIEDS) {
                $action = __(' posted a new ad', 'peepso-wpadverts');
            }
        }

        return ($action);
    }



    /**
     * Change act_id on repost button act_id to follow parent's act_id.
     * @param array $options The default options per post
     * @return  array
     */
    public function modify_post_actions($options)
    {
        $post = $options['post'];

        if (self::MODULE_ID === intval($post->act_module_id)) {

            $ads_type = get_post_meta($post->ID, self::POST_META_KEY_WPADVERTS_TYPE, true);
            if($ads_type === self::POST_META_KEY_WPADVERTS_TYPE_CLASSIFIEDS) {
                // disable repost function for classifieds post
                unset($options['acts']['repost']);
            }
        }

        return ($options);
    }

    /**
     * Render classifieds in user profile
     */
    public function action_profile_segment_classifieds()
    {
        $pro = PeepSoProfileShortcode::get_instance();
        $this->view_user_id = PeepSoUrlSegments::get_view_id($pro->get_view_user_id());

        $url = PeepSoUrlSegments::get_instance();
        if ($url->get(3) == 'manage') {

            $shortcode = PeepSoWPAdvertsShortcode::get_instance();
            echo $shortcode->shortcode_manage_ads();

        } else {

            wp_enqueue_script('peepso-wpdverts-classifieds');

            echo PeepSoTemplate::exec_template('wpadverts', 'profile-wpadverts', array('view_user_id' => $this->view_user_id), TRUE);
        }
    }

    /*
     * PeepSo navigation
     */

    public function filter_peepso_navigation($navigation)
    {
        $user = PeepSoUser::get_instance(get_current_user_id());

        $navigation['wpadverts'] = array(
            'href' => PeepSo::get_page('wpadverts'),
            'label' => PeepSo::get_option('wpadverts_navigation_label', __('Classifieds', 'peepso-wpadverts'), TRUE),
            'icon' => 'gcis gci-bullhorn',

            'primary'           => TRUE,
            'secondary'         => FALSE,
            'mobile-primary'    => TRUE,
            'mobile-secondary'  => FALSE,
            'widget'            => TRUE,
        );

        return ($navigation);
    }

    /**
     * Profile Segments - add link
     * @param $links
     * @return mixed
     */
    public function filter_peepso_navigation_profile($links)
    {
        $slug = PeepSo::get_option('wpadverts_navigation_profile_slug', 'classifieds', TRUE);

        $links[$slug] = array(
            'href' => $slug,
            'label'=> PeepSo::get_option('wpadverts_navigation_profile_label', __('Classifieds', 'peepso-wpadverts'), TRUE),
            'icon' => 'gcis gci-bullhorn'
        );

        return $links;
    }

    public static function isVersion140() {

        require_once(ABSPATH.'wp-admin/includes/plugin.php');

        $helloReflection = new ReflectionFunction('adverts_init');
        $plugin_data = get_plugin_data( $helloReflection->getFilename() );
        $plugin_version = isset($plugin_data['Version']) ? $plugin_data['Version'] : '';

        if (version_compare($plugin_version, '1.4.0', '>=')) {
            return true;
        }

        return false;
    }


    function custom_list_template( $tpl ) {

        $prefix = '';
        if (self::isVersion140()) {
            $prefix = '140';
        }

        // $tpl is an absolute path to a file, for example
        // /home/simpliko/public_html/wp-content/plugins/wpadverts/templates/list.php

        $basename = basename( $tpl );
        // $basename is just a filename for example list.php

        if( $basename == "add.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/add.php";
        } elseif( $basename == "add-preview.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/add-preview.php";
        } elseif( $basename == "add-save.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/add-save.php";
        } elseif( $basename == "manage.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/manage.php";
        } elseif( $basename == "manage-edit.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/manage-edit.php";
        } elseif( $basename == "single.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/single.php";
        } elseif( $basename == "categories-top.php" ) {
            return dirname( __FILE__ ) . "/templates/overrides" . $prefix . "/categories-top.php";
        } else {
            return $tpl;
        }
    }

    public function post_filters($options) {
        if (self::MODULE_ID == intval($options['post']->act_module_id)) {
            if (isset($options['acts']['edit'])) {
                unset($options['acts']['edit']);
            }
        }

        return $options;
    }
}

PeepSoWPAdverts::get_instance();
