<?php
/**
 * Plugin Name: PeepSo Integrations: Social Login & Invitations
 * Plugin URI: https://www.PeepSo.com
 * Description: Social Login & Invitations brought to you by the PeepSo team
 * Author: PeepSo
 * Version: 7.0.1.0
 * Copyright: (c) PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later (PHP), property of PeepSo, Inc. (CSS, JavaScript, images)
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: social-login
 * Domain Path: /language
 *
 * The PHP code of this plugin is Open Source. This PHP software can be redistributed and/or modified under the terms of the 
 * GNU General Public License (version 2 or later), as published by the Free Software Foundation. 
 * See the GNU General Public License or the LICENSE file for more details.  
 *  
 * Unless explicitly stated otherwise inside the respective files, all CSS, JavaScript, and images included in this plugin are proprietary to PeepSo, Inc.  
 * They are protected by copyright and other applicable intellectual property laws. 
 * Unauthorized use, reproduction, or distribution of these assets is STRICTLY PROHIBITED.  
 *  
 * "PeepSo" and "Your Community. Your Way." are registered trademarks of PeepSo, Inc.  
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY. 
*/

if(!class_exists('PeepSoSocialLoginPlugin')) {
    class PeepSoSocialLoginPlugin
    {
        private static $instance;
        public $path;

        const OPTION = 'social_login_option';
        const OPTION_HAS_PROVIDER = 'social_login_has_provider';

        public static function get_instance()
        {
            if (NULL === self::$instance) {
                self::$instance = new self();
            }
            return (self::$instance);
        }

        // PeepSo
        const SHORTCODE = 'peepso_social_invite';
        const PLUGIN_NAME = 'Integrations: Social Login & Invitations';
        const PLUGIN_VERSION = '7.0.1.0';
        const PLUGIN_RELEASE = ''; //ALPHA1, RC1 etc, '' for STABLE
        const PLUGIN_EDD = 32355890;
        const PLUGIN_SLUG = 'social-login';

        public $user_profile_model;

        public static $default_colors = [
            'Amazon'        => ['icon-color' => '#fff', 'background-color' => '#231f20'],
            'Apple'         => ['icon-color' => '#fff', 'background-color' => '#231f20'],
            'BitBucket'     => ['icon-color' => '#fff', 'background-color' => '#0052cc'],
            'Discord'       => ['icon-color' => '#fff', 'background-color' => '#7289da'],
            'Dropbox'       => ['icon-color' => '#fff', 'background-color' => '#0061ff'],
            'Facebook'      => ['icon-color' => '#fff', 'background-color' => '#3b5998'],
            'Foursquare'    => ['icon-color' => '#fff', 'background-color' => '#f94776'],
            'GitHub'        => ['icon-color' => '#fff', 'background-color' => '#24292e'],
            'GitLab'        => ['icon-color' => '#fff', 'background-color' => '#e65328'],
            'Google'        => ['icon-color' => 'rgba(0,0,0,0.54)', 'background-color' => '#fff'],
            'Instagram'     => ['icon-color' => '#fff', 'background-color' => '#517fa4'],
            'LinkedIn'      => ['icon-color' => '#fff', 'background-color' => '#0077b5'],
            'Reddit'        => ['icon-color' => '#fff', 'background-color' => '#ff5f24'],
            'Slack'         => ['icon-color' => '#fff', 'background-color' => '#511752'],
            'Steam'         => ['icon-color' => '#fff', 'background-color' => '#1b2838'],
            'Strava'        => ['icon-color' => '#fff', 'background-color' => '#fd4c06'],
            'Telegram'      => ['icon-color' => '#fff', 'background-color' => '#0088cc'],
            'Tumblr'        => ['icon-color' => '#fff', 'background-color' => '#35465c'],
            'TwitchTV'      => ['icon-color' => '#fff', 'background-color' => '#9147ff'],
            'Twitter'       => ['icon-color' => '#fff', 'background-color' => '#55acee'],
            'WeChat'        => ['icon-color' => '#fff', 'background-color' => '#2dc100'],
            'WeChatChina'   => ['icon-color' => '#fff', 'background-color' => '#2dc100'],
            'Apple'         => ['icon-color' => '#fff', 'background-color' => '#000000'],
        ];

        private static function ready() {
            if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION)) {
                return TRUE;
            }
        }

        private function __construct()
        {
            add_action('init', [$this, 'load_textdomain']);

            if(is_admin()) {

                register_activation_hook(__FILE__, [$this, 'activate']);

                add_action('admin_init', array(&$this, 'peepso_check'));

                add_filter('peepso_license_config', function ($list) {
                    $data = array(
                        'plugin_slug' => self::PLUGIN_SLUG,
                        'plugin_name' => self::PLUGIN_NAME,
                        'plugin_edd' => self::PLUGIN_EDD,
                        'plugin_version' => self::PLUGIN_VERSION
                    );
                    $list[] = $data;
                    return ($list);
                });

                add_filter('peepso_all_plugins', function ($plugins) {
                    $plugins[plugin_basename(__FILE__)] = get_class($this);
                    return $plugins;
                });
            }

            if(self::ready()) {

                if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
                    return;
                }

                $this->autoload();

                $this->path = plugin_dir_path(__FILE__);

                // 5423 add PeepSo overrides to SLAI template engine
                Social_Login_Template::add_template_directory(PeepSo::get_peepso_dir() . PeepSo::OVERRIDES_DIRECTORY . DIRECTORY_SEPARATOR);
                Social_Login_Template::add_template_directory(dirname(__FILE__));

                $this->user_profile_model = new Social_Login_User_Profile_Model();

                if (is_admin()) {

                    add_filter('peepso_admin_config_tabs', function($tabs){
                        $tabs['slai'] = array(
                            'label' => __('Social Login', 'groupso'),
                            'icon' => 'https://cdn.peepso.com/icons/plugins/'.self::PLUGIN_EDD.'.svg',
                            'tab' => 'slai',
                            'description' => __('PeepSo Social Login & Invitations', 'groupso'),
                            'function' => 'PeepSoConfigSectionSLAI',
                            'cat' => 'integrations',
                        );

                        return $tabs;
                    });
                    add_filter('peepso_admin_manage_tabs', function($tabs){
                        $tabs['slai'] = array(
                            'label' => __('Social Login', 'groupso'),
                            'icon' => 'https://cdn.peepso.com/icons/plugins/'.self::PLUGIN_EDD.'.svg',
                            'tab' => 'slai',
                            'description' => '',
                            'function' => array('Social_Login_New_Admin_Config', 'index'),
                            'cat'   => 'core',
                        );

                        return $tabs;
                    });
                } else {

                    $option = get_option(PeepSoSocialLoginPlugin::OPTION);

                    // show social login icon under PeepSo login form
                    if (PeepSo::get_option_new('slai_enabled')) {
                        add_action('peepso_action_render_login_form_after', function () {
                            $this->login_form([
                                'no_title' => TRUE
                            ]);
                        });

                        add_action('peepso_after_registration_form', function () {
                            $this->login_form([
                                'no_title' => TRUE
                            ]);
                        });

                        add_action('social_login_after_register', function ($user_id, $profile) use ($option) {
                            // get mapping field
                            $field_keys = [
                                'slai_first',
                                'slai_last',
                                'slai_bday',
                            ];
                            foreach ($field_keys as $key) {
                                $value = PeepSo::get_option($key);
                                if(!$value) continue;

                                $field = PeepSoField::get_field_by_id($value, $user_id);

                                if ($field) {
                                    // create new field
                                    switch ($key) {
                                        case 'slai_first':
                                            $field->save($profile->firstName);
                                            break;
                                        case 'slai_last':
                                            $field->save($profile->lastName);
                                            break;
                                        case 'slai_bday':
                                            if ($profile->birthDay != 0 && $profile->birthMonth != 0 && $profile->birthYear != 0) {
                                                $field->save($profile->birthYear . '-' . str_pad($profile->birthMonth, 2, '0', STR_PAD_LEFT) . '-' . str_pad($profile->birthDay, 2, '0', STR_PAD_LEFT));
                                            }
                                            break;
                                    }
                                }

                            }

                            $peepso_user = PeepSoUser::get_instance($user_id);

                            // if avatar is available
                            if (PeepSo::get_option('slai_avatars')) {
                                $result = wp_remote_get( # Not a critical PeepSo.com call, DON'T use PeepSoCom_Connect()
                                    $profile->photoURL, [
                                    'sslverify' => false,
                                ]);
                                if (!is_wp_error($result)) {
                                    $avatar = wp_remote_retrieve_body($result);
                                    $avatar_hash = substr(md5(time()), 0, 10);
                                    $path = PeepSo::get_peepso_dir() . '/users/' . $user_id;
                                    @mkdir($path);
                                    file_put_contents($path . '/' . $avatar_hash . '-avatar-full.jpg', $avatar);
                                    file_put_contents($path . '/' . $avatar_hash . '-avatar-orig.jpg', $avatar);
                                    file_put_contents($path . '/' . $avatar_hash . '-avatar.jpg', $avatar);
                                    add_user_meta($user_id, 'peepso_avatar_hash', $avatar_hash);
                                    $peepso_user->set_avatar_custom_flag();
                                }
                            }

                            if (isset($option['admin_verification']) && $option['admin_verification'] == 1 && PeepSo::get_option('site_registration_enableverification', 0) == 1) {
                                PeepSoRegisterShortcode::get_instance()->admin_approval($peepso_user);
                                nocache_headers();
                                wp_redirect(PeepSo::get_page('register') . '?success');
                                exit();
                            }
                        }, 10, 2);



                        add_filter('peepso_user_can_change_password', function ($allowed) {
                            $profile = $this->user_profile_model->get(['user_id' => get_current_user_id(), 'single' => TRUE]);
                            if ($profile) {
                                $allowed = FALSE;
                            }

                            return $allowed;
                        });

                        add_filter('peepso_edit_form_fields', function ($fields) {
                            $profile = $this->user_profile_model->get(['user_id' => get_current_user_id(), 'single' => TRUE]);
                            if ($profile) {
                                $fields['verify_password']['type'] = 'html';
                                $fields['verify_password']['html'] = '';
                                $fields['verify_password']['descript'] = sprintf(__('You are currently using <b>%s</b> to log in. <a onclick="return confirm(\'' . __('Are you sure?', 'social-login') . '\')" href="' . PeepSo::get_page('profile') . '?social-login-disconnect&nonce=' . wp_create_nonce('social-login-disconnect') . '">Disconnect</a>', 'social-login'), $profile->provider);
                            }

                            return $fields;
                        });

                        if (isset($_GET['social-login-disconnect'])) {
                            require_once(ABSPATH . 'wp-includes/pluggable.php');

                            if (wp_verify_nonce($_GET['nonce'], 'social-login-disconnect')) {
                                $current_user = wp_get_current_user();
                                $this->user_profile_model->delete($current_user->ID);

                                add_action('wp_logout', function () use ($current_user) {
                                    session_start();
                                    Social_Login_Helper::redirect(esc_url(site_url('password-recover?email=' . $current_user->user_email . '&nonce=' . md5(session_id()))));
                                    exit();
                                });

                                wp_logout();
                            }
                        }
                    }

                    add_filter('peepso_user_can_change_password', function ($allowed) {
                        $profile = $this->user_profile_model->get(['user_id' => get_current_user_id(), 'single' => TRUE]);
                        if ($profile) {
                            $allowed = FALSE;
                        }

                        return $allowed;
                    });

                    add_action('peepso_forgot_password_info', function () {
                        echo '<p>' . __('If you used a Social Login feature, it will be disconnected upon resetting the password.', 'social-login') . '</p>';
                    });

                    add_action('peepso_after_reset_password', function ($user) {
                        if ($user) {
                            $this->user_profile_model->delete($user->ID);
                        }
                    });

                    // get transient for error login
                    if (isset($_COOKIE['PHPSESSID'])) {
                        $transient = get_transient('error_login_' . $_COOKIE['PHPSESSID']);
                        if ($transient) {
                            add_action('wp_footer', function () use ($transient) {
                                wp_add_inline_script('social-login-error', 'alert("' . esc_attr($transient) . '");');
                            });
                            delete_transient('error_login_' . $_COOKIE['PHPSESSID']);
                        }
                    }
                }

                add_action('plugins_loaded', [$this, 'init']);

                add_action('init', [$this, 'register_block']);
            }


        }


        public function peepso_check()
        {
            if (!class_exists('PeepSo')) {
                add_action('admin_notices', function(){
                    ?>
                    <div class="error peepso">
                        <strong>
                            <?php echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'msgso'), self::PLUGIN_NAME);?>
                            <a href="plugin-install.php?tab=plugin-information&plugin=peepso-core&TB_iframe=true&width=772&height=291" class="thickbox">
                                <?php echo __('Get it now!', 'msgso');?>
                            </a>
                        </strong>
                    </div>
                    <?php
                });
                unset($_GET['activate']);
                deactivate_plugins(plugin_basename(__FILE__));
                return (FALSE);
            }

            // PeepSo.com license check
            if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
                add_action('admin_notices', function () {
                    PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG);
                });
            }

            if (isset($_GET['page']) && 'peepso_config' == sanitize_text_field($_GET['page']) && !isset($_GET['tab'])) {
                add_action('admin_notices', function () {
                    PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG, true);
                });
            }

            PeepSoLicense::check_updates_new(self::PLUGIN_EDD, self::PLUGIN_SLUG, self::PLUGIN_VERSION, __FILE__);

            return (TRUE);
        }

        private function autoload($path = NULL)
        {
            $class_paths = [];

            if (is_null($path)) {
                $class_paths[] = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'classes';
            } else {
                $class_paths[] = $path;
            }

            // $class_paths = apply_filters() // we will use filters later
            if (count($class_paths)) {
                foreach ($class_paths as $class_path) {
                    if (file_exists($class_path) && is_dir($class_path)) {
                        foreach (scandir($class_path) as $file) {
                            $path = $class_path . DIRECTORY_SEPARATOR . $file;
                            if (!is_dir($path)) {
                                require_once($path);
                            } else if ($file != '.' && $file != '..') {
                                $this->autoload($path);
                            }
                        }
                    }
                }
            }
        }

        public function init()
        {
            if (is_admin()) {
                add_action('admin_enqueue_scripts', function () {
                    wp_register_style('social-login', plugin_dir_url(__FILE__) . 'assets/css/sociallogin.css');
                    wp_enqueue_style('social-login');

                    // fix icon in backend
                    $page = isset($_GET['page'])? $_GET['page'] : '';
                    if(class_exists('PeepSo') && substr($page, 0,6) == 'peepso') {
                        wp_enqueue_style('ace-admin-boostrap-min');
                        wp_enqueue_style('ace-admin');
                        wp_enqueue_style('ace-admin-fontawesome');
                        wp_enqueue_style('peepso-admin');
                    }

                    wp_enqueue_style('wp-color-picker');
                    wp_register_script('wp-color-picker-alpha', plugin_dir_url(__FILE__) . 'assets/js/wp-color-picker-alpha.min.js', ['wp-color-picker'], FALSE, TRUE);
                    wp_enqueue_script('wp-color-picker-alpha');


                    wp_register_script('social-login-admin-config', plugin_dir_url(__FILE__) . 'assets/js/admin.js', ['jquery', 'jquery-ui-sortable', 'underscore', 'wp-color-picker-alpha'], self::PLUGIN_VERSION, TRUE);
                    wp_localize_script('social-login-admin-config', 'socialloginadmin', [
                        'ajaxurl' => admin_url('admin-ajax.php'),
                        'adminurl' => admin_url(),
                        'default_colors' => self::$default_colors,
                        'update_nonce' => wp_create_nonce('social-login-update')
                    ]);

                    wp_enqueue_script('social-login-admin-config');
                });
            }

            if (isset($_GET['loggedout'])) {
                $this->clear_user_session();
                // wp_clear_auth_cookie();
                @session_destroy();
            }

            add_action('login_form', [$this, 'login_form']);

            if (!wp_doing_ajax()) {
                add_action('init', [$this, 'process_login']);
            }

            add_action('widgets_init', function () {
                register_widget('Social_Login_Widget');
                register_widget('Social_Login_Invitation_Widget');
            });

            add_action('wp_logout', function () {
                $this->clear_user_session();
                // wp_clear_auth_cookie();
                @session_destroy();
            });

            if (is_admin()) {
                add_action('delete_user', function ($user_id) {
                    $this->user_profile_model->delete($user_id);
                });

                add_filter('wp_ajax_social_login_update', function() {
                    if (isset($_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'social-login-update') && current_user_can('administrator')) {
                        $prop = sanitize_text_field($_POST['prop']);

                        if ('box_status' == $prop) {
                            $status = sanitize_text_field($_POST['status']);

                            $id = json_decode(sanitize_text_field(stripslashes($_POST['id'])));

                            foreach ($id as $provider) {
                                update_user_meta(get_current_user_id(), 'social_login_open_' . $provider, $status);
                            }
                        }
                    }
                });

                add_filter('wp_ajax_social_login_reorder', function() {
                    if (current_user_can('administrator')) {
                        $option = get_option(PeepSoSocialLoginPlugin::OPTION);

                        $i = 0;
                        $ids = json_decode(sanitize_text_field(stripslashes($_POST['id'])));
                        foreach ($ids as $id) {
                            $option['providers'][$id]['order'] = $i;
                            $i++;
                        }

                        update_option(PeepSoSocialLoginPlugin::OPTION, $option);
                    }
                });
            }

            add_action('wp_enqueue_scripts', function () {
                wp_register_style('social-login', plugin_dir_url(__FILE__) . 'assets/css/sociallogin.css');
                wp_enqueue_style('social-login');
            });

            // Load CSS on WordPress login page
            add_action('login_enqueue_scripts', function () {
                wp_register_style('social-login', plugin_dir_url(__FILE__) . 'assets/css/sociallogin.css');
                wp_enqueue_style('social-login');
            });

            add_shortcode(self::SHORTCODE, [$this, 'shortcode']);

            if (!get_current_user_id()) {
                // WooCommerce checkout integration
                add_action('woocommerce_before_checkout_form', function() {
                    $this->login_form([
                        'no_title' => FALSE,
                        'template' => 'login-checkout'
                    ]);
                });

                // EDD checkout integration
                add_action('edd_purchase_form_top', function() {
                    $this->login_form([
                        'no_title' => FALSE,
                        'template' => 'login-checkout'
                    ]);
                });
            }
        }

        public function activate()
        {
            // create table
            global $wpdb;
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

            $wpdb->suppress_errors(TRUE);
            $wpdb->query("RENAME TABLE {$wpdb->prefix}social_login_user_profile TO {$wpdb->prefix}peepso_social_login_user_profile");
            $wpdb->suppress_errors(FALSE);

            $version = get_option('PeepSo_Social_Login', '3.0');
            $charset_collate = $wpdb->get_charset_collate();

            $sql = "CREATE TABLE " . $wpdb->prefix . "peepso_social_login_user_profile (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id int(11) NOT NULL,
			provider varchar(50) NOT NULL,
            email varchar(255) NOT NULL,
			emailverified varchar(255) NULL,
			identifier varchar(255) NULL,
			profileurl varchar(255) NULL,
			websiteurl varchar(255) NULL,
			photourl varchar(255) NULL,
			displayname varchar(150) NULL,
			description varchar(255) NULL,
			firstname varchar(150) NULL,
			lastname varchar(150) NULL,
			gender varchar(10) NULL,
			language varchar(20) NULL,
			age varchar(10) NULL,
			job_title varchar(255) NULL,
			organization_name varchar(255) NULL,
			birthday int(11) NULL,
			birthmonth int(11) NULL,
			birthyear int(11) NULL,
			phone varchar(75) NULL,
			address varchar(255) NULL,
			country varchar(75) NULL,
			region varchar(50) NULL,
			city varchar(50) NULL,
			zip varchar(25) NULL,
            PRIMARY KEY (id)
		) $charset_collate;";

            dbDelta($sql);
        }

        public function load_textdomain()
        {
            $path = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'language' . DIRECTORY_SEPARATOR . 'social-login-' . determine_locale() . '.mo';
            load_textdomain('social-login', $path);
        }

        public static function get_asset($path)
        {
            return plugin_dir_url(__FILE__) . 'assets/' . $path;
        }

        public function login_form($args)
        {
            // #5265 Elementor Pro activation compatibility
            if (isset($_GET['page']) && 'elementor-connect' == sanitize_text_field($_GET['page'])) {
                return;
            }

            if (!isset($args['template'])) {
                $template = 'social-login';
            } else {
                $template = $args['template'];
            }

            $enabled = get_option(self::OPTION_HAS_PROVIDER, FALSE);

            if ($enabled || $enabled === FALSE) { // FALSE means no option has been set
                $login = '';
                $providers = self::get_provider_option();
                foreach ($providers as $key => $value) {
                    $icon_color = !empty($value['icon-color']) ? 'color:' . $value['icon-color'] . ';' : '';
                    $background_color = !empty($value['background-color']) ? 'background-color:' . $value['background-color'] . ';' : '';

                    if ($value['enabled'] == 1) {
                        $url = site_url('wp-login.php', 'login_post') . (strpos(site_url('wp-login.php', 'login_post'), '?') ? '&' : '?' ) . "action=social_login_authenticate&provider=" . $key . '&mode=login';
                        if ($key == "Google") {
                            $login .= '<div class="tsc-icons__item"><a href="' . $url . '" class="tsc-icon tsc-icon--' . strtolower($key) . '" data-tooltip="'. $key .'" style="' . $icon_color . $background_color . '"><img src="' . plugin_dir_url(__FILE__) . '/assets/img/Google-SVG.svg" alt="Google" />' . $key . '</a></div>';
                        } else {
                            $login .= '<div class="tsc-icons__item"><a href="' . $url . '" class="tsc-icon tsc-icon--' . strtolower($key) . '" data-tooltip="'. $key .'" style="' . $icon_color . $background_color . '"><i class="tfab tfa-' . strtolower($key) . '"></i></a></div>';
                        }
                    }
                }

                if (!empty($login) && $enabled === FALSE) {
                    update_option(self::OPTION_HAS_PROVIDER, 1);
                } else if (empty($login)) {
                    return;
                }

                echo esc_html(Social_Login_Template::exec_template('general', $template, [
                    'no_title' => isset($args['no_title']) ? $args['no_title'] : FALSE,
                    'title' => isset($args['title']) ? esc_attr($args['title']) : '',
                    'login' => $login,
                    'widget_args' => isset($args['widget_args']) ? $args['widget_args'] : NULL,
                ]));
            }
        }

        public static function get_provider_option()
        {
            $option = get_option(self::OPTION, []);
            if (!isset($option['providers'])) {
                $option['providers'] = $option;
                update_option(self::OPTION, $option);
            }
            $option = $option['providers'];
            $provider_config_keys = [
                'Amazon'      => ['id', 'secret'],
                'Apple'       => ['id', 'team_id', 'key_id', 'key_content'],
                'BitBucket'   => ['id', 'secret'],
                'Discord'     => ['id', 'secret'],
                'Dropbox'     => ['id', 'secret'],
                'Facebook'      => ['id', 'secret'],
                'Foursquare'  => ['id', 'secret'],
                'GitHub'      => ['id', 'secret'],
                'GitLab'      => ['id', 'secret'],
                'Google'        => ['id', 'secret'],
                'Instagram'     => ['id', 'secret'],
                'LinkedIn'    => ['id', 'secret'],
                'Reddit'        => ['id', 'secret'],
                'Slack'       => ['id', 'secret'],
                'Steam'       => ['secret'],
                'Strava'      => ['id', 'secret'],
                'Telegram'    => ['id', 'secret'],
                'Tumblr'      => ['id', 'secret'],
                'TwitchTV'    => ['id', 'secret'],
                'Twitter'       => ['key', 'secret'],
                'WeChat'      => ['id', 'secret'],
                'WeChatChina' => ['id', 'secret'],
            ];

            $i = 0;
            foreach ($provider_config_keys as $provider => $keys) {
                $providers_option[$provider]['enabled'] = isset($option[$provider]['enabled']) ? $option[$provider]['enabled'] : 0;

                if (count($keys) > 0) {
                    $provider_keys = [];
                    foreach ($keys as $key) {
                        $provider_keys[$key] = isset($option[$provider][$key]) ? $option[$provider][$key] : '';
                    }

                    $providers_option[$provider]['keys'] = $provider_keys;
                }

                $providers_option[$provider]['icon-color'] = isset($option[$provider]['icon-color']) ? $option[$provider]['icon-color'] : self::$default_colors[$provider]['icon-color'];
                $providers_option[$provider]['background-color'] = isset($option[$provider]['background-color']) ? $option[$provider]['background-color'] : self::$default_colors[$provider]['background-color'];

                if (!isset($option[$provider]['order']) || isset($_GET['social_login_reset_order'])) {
                    $providers_option[$provider]['order'] = $i;
                    $i++;
                } else {
                    $providers_option[$provider]['order'] = $option[$provider]['order'];
                }
            }

            if (isset($_GET['social_login_reset_order'])) {
                $option = get_option(self::OPTION);
                $option['providers'] = $providers_option;
                update_option(PeepSoSocialLoginPlugin::OPTION, $option);
            }

            uasort($providers_option, function($a, $b) {
                return $a['order'] <=> $b['order'];
            });

            return $providers_option;
        }

        // login functions

        public function process_login()
        {
            // #5265 Elementor Pro activation compatibility
            if (isset($_GET['page']) && 'elementor-connect' == sanitize_text_field($_GET['page'])) {
                return FALSE;
            }

            $action = $this->process_login_get_param('action');
            $auth_mode = $this->process_login_get_param('mode');
            $redirect_to = $this->process_login_get_param('redirect_to');
            $provider = $this->process_login_get_param('provider');
            $code = $this->process_login_get_param('code');

            if ((!in_array($action, ['social_login_authenticate', 'social_login_account_create', 'social_login_account_linking', 'social_login_authenticated']) && !isset($code)) ||
                (isset($code) && empty($action) && (($provider != 'Instagram' && $provider != 'Strava') || isset($_SERVER['HTTP_REFERER']) && strpos($_SERVER['HTTP_REFERER'], site_url()) !== FALSE))) {
                return false;
            }

            add_action('login_enqueue_scripts', function() {
                wp_dequeue_style('login');
                wp_enqueue_script('social-login-script', plugin_dir_url(__FILE__) . 'assets/js/login.js');
                wp_enqueue_style('social-login-style', plugin_dir_url(__FILE__) . 'assets/css/login.css');
                wp_enqueue_style('social-login', plugin_dir_url(__FILE__) . 'assets/css/sociallogin.css');
            });

            $error_message = '';

            if ($action == 'social_login_authenticate') {
                return $this->process_login_begin();
            } else if ($action == 'social_login_account_create') {
                return $this->create_account();
            } else if ($action == 'social_login_account_linking') {
                $error_message = $this->link_account();
            }

            $hybridauth_profile = $this->get_social_user_profile();

            if ($hybridauth_profile) {
                // check if user exist
                $profile = $this->user_profile_model->get(['identifier' => $hybridauth_profile->identifier, 'provider' => $provider, 'single' => TRUE]);
                if (!is_object($profile)) {
                    // if no, show gateway page
                    if (empty($hybridauth_profile->photoURL)) {
                        $hybridauth_profile->photoURL = plugin_dir_url(__FILE__) . '/assets/img/default-avatar.png';
                    }

                    add_action('login_header', function() use($provider, $redirect_to, $hybridauth_profile, $error_message) {
                        echo esc_html(Social_Login_Template::exec_template('general', 'gateway', [
                            'provider' => $provider,
                            'redirect_to' => $redirect_to,
                            'hybridauth_user_profile' => $hybridauth_profile,
                            'error_message' => $error_message,
                        ]));
                        die();
                    });

                } else {
                    // proceed login
                    $user_profile = $this->user_profile_model->get(['single' => TRUE, 'identifier' => $hybridauth_profile->identifier, 'provider' => $provider]);
                    $this->login_as_user($user_profile->user_id);
                }
            }
        }


        public function process_login_begin()
        {
            $provider = $this->process_login_get_param('provider');
            $auth_mode = $this->process_login_get_param('mode');

            if (is_null($provider)) {
                return;
            }

            if (!isset($_REQUEST['redirect_to_provider'])) {
                $this->clear_user_session();
                @session_start();
                session_destroy();
                $_SESSION['social_login_last_visited_page'] = $_SERVER['HTTP_REFERER'];

                add_action('login_header', function() use($provider) {
                    echo esc_html(Social_Login_Template::exec_template('general', 'loading', [
                        'provider' => $provider,
                        'mode' => 'redirect'
                    ]));
                    die();
                });
            }

            $hybridauth_profile = $this->get_social_user_profile();

            if ($hybridauth_profile) {
                $authenticated_url = site_url('wp-login.php', 'login_post') . (strpos(site_url('wp-login.php', 'login_post'), '?') ? '&' : '?') . 'action=social_login_authenticated&provider=' . $provider . '&mode=' . $auth_mode;

                add_action('login_header', function() use($provider, $authenticated_url) {
                    echo esc_html(Social_Login_Template::exec_template('general', 'loading', [
                        'provider' => $provider,
                        'mode' => 'process',
                        'authenticated_url' => $authenticated_url,
                        'redirect_to' => $this->process_login_get_param('redirect_to', home_url()),
                    ]));
                    die();
                });
            }
        }

        public function get_hybridauth_config($provider)
        {
            $option = get_option(self::OPTION);
            $option = $option['providers'];

            if (in_array($provider, ['Instagram', 'Strava'])) {
                $callback = site_url('wp-login.php');
            } else {
                $callback = site_url('wp-login.php?action=social_login_authenticated&provider=' . $provider);
            }

            $config = [
                'callback' => $callback,
                'providers' => self::get_provider_option()
            ];

            return $config;
        }

        public function load_hybridauth()
        {
            if (!class_exists('HybridauthSO', false)) {
                require_once $this->path . 'lib/autoload.php';
            }
        }

        public function create_account()
        {
            $hybridauth_profile = $this->get_social_user_profile();

            if ($hybridauth_profile) {
                if (empty($hybridauth_profile->email) || isset($_POST['email'])) {
                    $hybridauth_profile->email = sanitize_email($_POST['email']);
                }

                $option = get_option(self::OPTION);
                $error_message = '';

                if (PeepSo::get_option('slai_enabled') && !empty($hybridauth_profile->email)) {
                    $valid_email = apply_filters('peepso_register_valid_email', TRUE, $hybridauth_profile->email);

                    if (!$valid_email) {
                        $error_message = sprintf(__('%s is blocked, please use another email address.', 'social-login'), $hybridauth_profile->email);
                    }
                }

                if (!$error_message) {
                    // create new user profile
                    $username = str_replace(' ', '_', strtolower($hybridauth_profile->displayName));
                    if (empty($username)) {
                        $username = preg_replace('/[\W]/', '', $hybridauth_profile->email);
                    }
                    if (username_exists($username)) {
                        $username .= rand(9, 9999);
                    }
                    $user_id = wp_create_user($username, wp_generate_password(12, FALSE), $hybridauth_profile->email);
                    if (is_object($user_id)) {
                        $error_message = $user_id->get_error_message();
                    }
                }

                if ($error_message) {
                    add_action('login_header', function() use($error_message) {
                        echo esc_html(Social_Login_Template::exec_template('general', 'loading', [
                            'mode' => 'error',
                            'error_message' => $error_message
                        ]));
                        die();
                    });
                } else {
                    $hybridauth_profile = $this->create_social_user_profile($user_id);
                    if ($hybridauth_profile) {
                        do_action('social_login_after_register', $user_id, $hybridauth_profile);
                        $this->login_as_user($user_id);
                    }
                }
            }
        }

        public function link_account()
        {
            // check if user exist
            $user = wp_signon([
                'user_login' => sanitize_text_field($_POST['username']),
                'user_password' => $_POST['password']
            ]);

            if (isset($user->ID)) {
                $hybridauth_profile = $this->create_social_user_profile($user->ID);
                if ($hybridauth_profile) {
                    do_action('social_login_after_register', $user->ID, $hybridauth_profile);
                    $this->login_as_user($user->ID);
                }
            } else {
                return __('Invalid username or password', 'social-login');
            }
        }

        public function login_as_user($user_id)
        {
            global $wp_filter;

            // Two Factor Authentication compatibility
            foreach ($wp_filter['authenticate']->callbacks as $priority_key => $priority) {
                foreach ($priority as $callback_key => $callback) {
                    if (is_array($callback['function']) && $callback['function'][1] == 'tfaVerifyCodeAndUser') {
                        remove_filter('authenticate', [$callback['function'][0], $callback['function'][1]], $priority_key);
                    }
                }
            }

            add_filter('check_password', function ($check, $password, $hash, $user_id) {
                return TRUE;
            }, 99, 4);

            $redirect = site_url();

            $user = get_user_by('id', $user_id);
            $info['user_login'] = $user->user_login;
            $info['user_password'] = rand(0, 10);

            $user_signon = wp_signon($info, TRUE);
            if (!is_wp_error($user_signon)) {
                $redirect = PeepSo::get_page('redirectlogin');
                if (empty($redirect)) {
                    $redirect = PeepSo::get_page('profile');
                }
            } else {
                set_transient('error_login_' . $_COOKIE['PHPSESSID'], $user_signon->get_error_message());
            }

            nocache_headers();
            wp_redirect($redirect);
            exit();
        }

        public function get_social_user_profile()
        {
            $provider = $this->process_login_get_param('provider');
            $transient_key = 'social_login_' . $_COOKIE['PHPSESSID'] . '_' . $provider;
            $hybridauth_profile = get_transient($transient_key);

            if (empty($hybridauth_profile)) {
                $this->load_hybridauth();

                try {
                    $hybridauth = new HybridauthSO\HybridauthSO($this->get_hybridauth_config($provider));
                    $adapter = $hybridauth->authenticate($provider);
                } catch (Exception $e) {
                    // show error message
                    add_action('login_header', function() use($e) {
                        echo esc_html(Social_Login_Template::exec_template('general', 'loading', [
                            'mode' => 'error',
                            'error_message' => $e->getMessage()
                        ]));
                        die();
                    });
                }

                if (isset($adapter)) {
                    $hybridauth_profile = json_decode(json_encode($adapter->getUserProfile()));
                    set_transient($transient_key, $hybridauth_profile, 600);
                }
            }

            return $hybridauth_profile;
        }

        public function create_social_user_profile($user_id)
        {
            $provider = $this->process_login_get_param('provider');
            $hybridauth_profile = $this->get_social_user_profile();
            if ($hybridauth_profile) {
                $user_data = [
                    'provider' => $provider,
                    'user_id' => $user_id
                ];

                foreach (get_object_vars($hybridauth_profile) as $key => $value) {
                    $user_data[strtolower($key)] = $value;
                }

                unset($user_data['data']);
                $user_profile_id = $this->user_profile_model->create($user_data);


                // Due to #6019 and since 4.0.5.0 this actually stores the ID
                // Before 4.0.5.0  it stored results of $wpdb->insert() so be careful when handling pre-4.0.5.0 data
                add_user_meta($user_id, 'social_login_profile_id', $user_profile_id);
            }

            return $hybridauth_profile;
        }

        private function process_login_get_param($param, $default = null)
        {
            if ($param == 'provider') {
                if (isset($_SERVER['HTTP_REFERER'])) {
                    if (strpos($_SERVER['HTTP_REFERER'], 'instagram.com')) {
                        $default = 'Instagram';
                    } else if (strpos($_SERVER['HTTP_REFERER'], 'strava.com')) {
                        $default = 'Strava';
                    }
                } else {
                    $default = 'Instagram';
                }
            } else if ($param == 'mode') {
                $default = 'login';
            }
            return isset($_REQUEST[$param]) ? sanitize_text_field($_REQUEST[$param]) : $default;
        }

        private function clear_user_session()
        {
            $_SESSION['HA::STORE'] = []; // used by hybridauth library. to clear as soon as the auth process ends.
            $_SESSION['HA::CONFIG'] = []; // used by hybridauth library. to clear as soon as the auth process ends.
            $_SESSION['tp::userprofile'] = []; // used by wsl to temporarily store the user profile so we don't make unnecessary calls to social apis.
        }

        public function register_block()
        {
            if (!function_exists('register_block_type')) {
                // Gutenberg is not active.
                return;
            }

            // register block script
            wp_register_script(
                'social-login/social-login',
                plugins_url('assets/js/block.js', __FILE__),
                ['wp-blocks', 'wp-i18n', 'wp-element', 'wp-editor', 'underscore'],
                filemtime(plugin_dir_path(__FILE__) . 'assets/js/block.js')
            );

            wp_localize_script('social-login/social-login', 'sociallogin', [
                'plugin_name' => __('Social Invitation', 'social-login'),
                'site_url' => site_url()
            ]);

            global $pagenow;
            if ($pagenow != 'widgets.php') {
                // register block
                register_block_type('social-login/social-login', [
                        'render_callback' => [$this, 'block'],
                        'editor_script' => 'social-login/social-login',
                        'style' => 'social-login/social-login',
                        'attributes' => [
                            'link' => [
                                'type' => 'string'
                            ]
                        ]
                    ]
                );
            }
        }

        public function block($attributes, $content)
        {
            if (isset($_GET['context']) && sanitize_text_field($_GET['context']) == 'edit') {
                // editor page render
                $attributes['preview'] = 'block';
            }

            $attributes['class'] = 'tp-sl-block';
            $output = $this->render_output($attributes);

            return $output;
        }

        public function shortcode($attributes)
        {
            $attributes['class'] = 'tp-sl-shortcode';
            $output = $this->render_output($attributes);

            return $output;
        }

        public function render_output($attributes, $args = [])
        {
            $option = self::get_provider_option();

            $attributes['facebook_app_id'] = $option['Facebook']['keys']['id'];
            $url = isset($attributes['link']) ? esc_url($attributes['link']) : home_url();

            wp_enqueue_script('facebook-sdk', 'https://connect.facebook.net/en_US/sdk.js');
            wp_add_inline_script('facebook-sdk', 'window.fbAsyncInit=function(){FB.init({appId:"' . $attributes['facebook_app_id'] . '",autoLogAppEvents:!0,xfbml:!0,version:"v4.0"})};');
            wp_add_inline_script('facebook-sdk', 'jQuery(".share-facebook").on("click", function(){FB.ui({method:"send",link:"' . $url . '"})});');

            $invitation = Social_Login_Template::exec_template('general', 'invite', ['instance' => $attributes, 'args' => $args], TRUE);
            return $invitation;
        }
    }

    PeepSoSocialLoginPlugin::get_instance();
}
