import $ from 'jquery';
import { observer, template } from 'peepso';
import peepsodata from 'peepsodata';
import { rsvp } from './ajax';

const wpem = window.peepso_wpem;

const USER_ID = +peepsodata.currentuserid;
const RSVP_STATES = (wpem.data && wpem.data.rsvp_states) || {};
const RSVP_DROPDOWN = (wpem.tmpl && wpem.tmpl.rsvp_dropdown_stream) || '';

/**
 * Initialize activity item.
 *
 * @param {Element} elem
 */
function initItem(elem) {
	let $post = $(elem);
	let $toggle = $post.find('.ps-js-rsvp-toggle');
	let id = $toggle.data('event-id');
	let state = 'no';

	if (!$toggle.length) {
		return;
	}

	if (!USER_ID) {
		updateButton($toggle, state);
		return;
	}

	updateButton($toggle, state, 'loading');
	rsvp(id)
		.then(json => initDropdown($toggle, (state = json.status)))
		.catch(error => error && alert(error))
		.finally(() => updateButton($toggle, state));
}

/**
 * Initialize RSVP dropdown.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 */
function initDropdown($toggle, state = 'no') {
	let id = $toggle.data('event-id');
	let $dropdown = $(template(RSVP_DROPDOWN)());
	let $options = $dropdown.find('button[data-state]');

	$dropdown.hide();
	$dropdown.on('click', function (e) {
		e.preventDefault();
		e.stopPropagation();

		$dropdown.hide();
	});

	$options.removeClass('active');
	$options.filter(`[data-state="${state}"]`).addClass('active');

	// Listen to global RSVP update.
	observer.addAction('wpem_rsvp_update', json => {
		if (+json.event_id && id == json.event_id) {
			state = json.status;

			updateButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		}
	});

	$dropdown.on('click', 'button', function (e) {
		let newState = $(this).data('state');
		if (newState === state) {
			return;
		}

		// Optimistic update.
		updateButton($toggle, newState, 'loading');
		// No need to update the button here as it will be handled by the `wpem_rsvp_update` action above.
		rsvp(id, newState)
			.then(json => (state = json.status))
			.catch(error => error && alert(error));
	});

	$toggle.css({ position: 'relative' });
	$toggle.append($dropdown);
	$toggle.on('click', () => $dropdown.toggle());
}

/**
 * Update RSVP label.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateButton($toggle, state = 'no', mode) {
	let icon = 'loading' === mode ? 'gci-spinner gci-spin' : RSVP_STATES[state].icon;
	let text = RSVP_STATES[state].label;

	// Update button label and icon.
	$toggle.children('i').attr('class', `gcis ${icon}`);
	$toggle.children('span').text(text);

	// Disable button on loading.
	$toggle.css('opacity', 'loading' === mode ? 0.3 : '');
}

// Initialize on each activity item added.
$(function () {
	observer.addFilter('peepso_activity', $posts =>
		$posts.map(function () {
			if (this.nodeType === 1) {
				initItem(this);
			}
			return this;
		})
	);
});
