import { ajax, observer } from 'peepso';
import { currentuserid, rest_url as REST_URL } from 'peepsodata';

const USER_ID = +currentuserid;

/**
 * Get or set RSVP status of a specific event.
 *
 * @param {number|string} id
 * @param {string|undefined} status
 * @returns {Promise.<Object,?string>}
 */
function rsvp(id, status) {
	return new Promise((resolve, reject) => {
		let endpoint = `${REST_URL}wpem_rsvp`,
			params = /^http/i.test(id) ? { event_url: id } : { event_id: id };

		function callback(data) {
			let error;

			if ('object' === typeof data) {
				if (['yes', 'no', 'maybe'].find(s => s === data.status)) {
					resolve(data);

					// Broadcast RSVP status update.
					if ('undefined' !== typeof status) {
						observer.doAction('wpem_rsvp_update', data);
					}

					return;
				}

				if (data.error) {
					error = data.error; // Endpoint error message.
				} else if (data.code && data.message) {
					error = data.message; // WordPress default REST error response format.
				}
			}

			if ('string' !== typeof error) {
				error = undefined;
			}

			reject(error);
		}

		if (!USER_ID) {
			callback();
		} else if ('undefined' === typeof status) {
			ajax.get(endpoint, params, 20).done(callback).fail(callback);
		} else {
			params.status = ['yes', 'maybe'].indexOf(status) >= 0 ? status : 'no';
			ajax.post(endpoint, params, 20).done(callback).fail(callback);
		}
	});
}

/**
 * Get or set RSVP notification status of a specific event.
 *
 * @param {number} id
 * @param {boolean|undefined} status
 * @returns {Promise.<Object,?string>}
 */
function rsvpNotifications(id, status) {
	return new Promise((resolve, reject) => {
		let endpoint = `${REST_URL}wpem_rsvp_notifications`,
			params = { event_id: id };

		function callback(data) {
			let error;

			if ('object' === typeof data) {
				if ([true, false].indexOf(data.status) >= 0) {
					resolve(data);

					// Broadcast RSVP notification status update.
					if ('undefined' !== typeof status) {
						observer.doAction('wpem_rsvp_notification_update', data);
					}

					return;
				}

				if (data.error) {
					error = data.error; // Endpoint error message.
				} else if (data.code && data.message) {
					error = data.message; // WordPress default REST error response format.
				}
			}

			reject(error);
		}

		if (!USER_ID) {
			callback();
		} else if ('undefined' === typeof status) {
			ajax.get(endpoint, params, 20).done(callback).fail(callback);
		} else {
			params.status = +status ? 1 : 0;
			ajax.post(endpoint, params, 20).done(callback).fail(callback);
		}
	});
}

export { rsvp, rsvpNotifications };
