import $ from 'jquery';
import { observer, template } from 'peepso';
import peepsodata from 'peepsodata';
import { rsvp } from './ajax';

const wpem = window.peepso_wpem;

const USER_ID = +peepsodata.currentuserid;
const RSVP_STATES = (wpem.data && wpem.data.rsvp_states) || {};
const RSVP_DROPDOWN = (wpem.tmpl && wpem.tmpl.rsvp_dropdown_widget) || '';

/**
 * Initialize event listing page view.
 *
 * @param {Element} container
 */
function initItems(container) {
	let items = [...container.querySelectorAll('.calendar-event-details-link')].filter(
		item => !item.className.match(/ps-js-initialized/)
	);

	items.forEach(item => {
		let $item = $(item).addClass('ps-js-initialized');
		let fadeOutTimer;

		// Improve toggle popup handler.
		$item
			.off('mouseover mouseout')
			.on('mouseenter', function (e) {
				clearTimeout(fadeOutTimer);
				$(this).children('div[id]').fadeIn('fast');
			})
			.on('mouseleave', function () {
				fadeOutTimer = setTimeout(() => {
					clearTimeout(fadeOutTimer);
					$(this).children('div[id]').fadeOut('fast');
				}, 300);
			});

		initRsvp(item);
	});
}

/**
 * Initialize RSVP dropdown toggle.
 *
 * @param {Element} item
 */
function initRsvp(item) {
	let $item = $(item);
	let $container = $item.find('.calendar-tooltip').css('overflow', 'visible');
	let $content = $container.find('.calendar-tooltip-content');
	let $dropdown = $(template(RSVP_DROPDOWN)({ event_id: 0 })).appendTo($content);
	if (!$dropdown.length) {
		return;
	}

	let $toggle = $dropdown.find('.ps-js-dropdown-toggle');
	let $options = $dropdown.find('button[data-state]');
	let id = $item.children('div[id]').attr('id').replace(/[^\d]/gi, '');
	let state = 'no';

	// Update event_id data on the dropdown.
	$dropdown.attr('data-event-id', id);

	// Listen to global RSVP update.
	observer.addAction('wpem_rsvp_update', json => {
		if (+json.event_id && id == json.event_id) {
			state = json.status;

			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		}
	});

	updateRsvpButton($toggle, state, 'loading');
	rsvp(id)
		.then(json => {
			state = json.status;

			$options.on('click', function (e) {
				e.preventDefault();
				e.stopPropagation();

				$toggle.trigger('click');

				let newState = $(this).data('state');
				if (newState === state) {
					return;
				}

				// Optimistic update.
				updateRsvpButton($toggle, newState, 'loading');
				// No need to update the button here as it will be handled by the `wpem_rsvp_update` action above.
				rsvp(id, newState)
					.then(json => (state = json.status))
					.catch(error => error && alert(error));
			});
		})
		.catch(error => error && alert(error))
		.finally(() => {
			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		});
}

/**
 * Update RSVP button label.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateRsvpButton($toggle, state = 'no', mode) {
	let icon = 'loading' === mode ? 'gci-spinner gci-spin' : RSVP_STATES[state].icon;
	let text = RSVP_STATES[state].label;

	// Update button label and icon.
	$toggle.children('i').attr('class', `gcis ${icon}`);
	$toggle.children('span').text(text);

	// Disable button on loading.
	$toggle.css('opacity', 'loading' === mode ? 0.3 : '');
	$toggle.prop('disabled', 'loading' === mode);
}

// Prefilter AJAX request parameters.
// https://api.jquery.com/jQuery.ajaxPrefilter/
$.ajaxPrefilter(function (options, originalOptions, jqXHR) {
	if (/^\/em-ajax\/events_calendar(_layout)?\//.test(options.url)) {
		let $form = $('form#event_filters');

		// Add form_data parameter.
		options.data = `${options.data}&form_data=${encodeURIComponent($form.serialize())}`;
	}
});

$(function () {
	let container = document.querySelector('.wpem-event-listings');
	if (!(container && USER_ID)) {
		return;
	}

	initItems(container);

	new MutationObserver(function (mutationsList) {
		for (var mutation of mutationsList) {
			if ('childList' === mutation.type) {
				initItems(container);
			}
		}
	}).observe(container, {
		subtree: true,
		childList: true,
		attributes: false,
		characterData: false
	});
});
