import $ from 'jquery';
import { observer } from 'peepso';
import peepsodata from 'peepsodata';
import { rsvp } from './ajax';

const wpem = window.peepso_wpem;

const USER_ID = +peepsodata.currentuserid;
const RSVP_STATES = (wpem.data && wpem.data.rsvp_states) || {};

/**
 * Initialize event listing page view.
 *
 * @param {Element} container
 */
function initItems(container) {
	let items = [...container.querySelectorAll('.wpem-event-box-col')].filter(
		item => !item.className.match(/ps-js-initialized/)
	);

	items.forEach(item => {
		$(item).addClass('ps-js-initialized');

		initRsvp(item);
	});
}

/**
 * Initialize RSVP dropdown toggle.
 *
 * @param {Element} item
 */
function initRsvp(item) {
	let $dropdown = $(item).find('.ps-js-dropdown');
	if (!$dropdown.length) {
		return;
	}

	let $toggle = $dropdown.find('.ps-js-dropdown-toggle');
	let $options = $dropdown.find('button[data-state]');
	let id = $dropdown.data('event-id');
	let state = 'no';

	// Listen to global RSVP update.
	observer.addAction('wpem_rsvp_update', json => {
		if (+json.event_id && id == json.event_id) {
			state = json.status;

			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		}
	});

	updateRsvpButton($toggle, state, 'loading');
	rsvp(id)
		.then(json => {
			state = json.status;

			$options.on('click', function (e) {
				e.preventDefault();

				let newState = $(this).data('state');
				if (newState === state) {
					return;
				}

				// Optimistic update.
				updateRsvpButton($toggle, newState, 'loading');
				// No need to update the button here as it will be handled by the `wpem_rsvp_update` action above.
				rsvp(id, newState)
					.then(json => (state = json.status))
					.catch(error => error && alert(error));
			});
		})
		.catch(error => error && alert(error))
		.finally(() => {
			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		});
}

/**
 * Update RSVP button label.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateRsvpButton($toggle, state = 'no', mode) {
	let icon = 'loading' === mode ? 'gci-spinner gci-spin' : RSVP_STATES[state].icon;
	let text = RSVP_STATES[state].label;

	// Update button label and icon.
	$toggle.children('i').attr('class', `gcis ${icon}`);
	$toggle.children('span').text(text);

	// Disable button on loading.
	$toggle.css('opacity', 'loading' === mode ? 0.3 : '');
	$toggle.prop('disabled', 'loading' === mode);
}

$(function () {
	let container = document.querySelector('.wpem-event-listings');
	if (!(container && USER_ID)) {
		return;
	}

	initItems(container);

	new MutationObserver(function (mutationsList) {
		for (var mutation of mutationsList) {
			if ('childList' === mutation.type) {
				initItems(container);
			}
		}
	}).observe(container, {
		subtree: false,
		childList: true,
		attributes: false,
		characterData: false
	});
});
