import $ from 'jquery';
import { observer } from 'peepso';
import peepsodata from 'peepsodata';
import { rsvp, rsvpNotifications } from './ajax';

const wpem = window.peepso_wpem;

const USER_ID = +peepsodata.currentuserid;
const RSVP_STATES = (wpem.data && wpem.data.rsvp_states) || {};

/**
 * Initialize single event page view.
 *
 * @param {Element} item
 */
function initItem(item) {
	$(item).addClass('ps-js-initialized');

	initRsvp(item);
	initRsvpNotifications(item);
}

/**
 * Initialize RSVP dropdown toggle.
 *
 * @param {Element} item
 */
function initRsvp(item) {
	let $dropdown = $(item).find('.ps-js-dropdown');
	if (!$dropdown.length) {
		return;
	}

	let $toggle = $dropdown.find('.ps-js-dropdown-toggle');
	let $options = $dropdown.find('button[data-state]');
	let id = $dropdown.data('event-id');
	let state = 'no';

	// Listen to global RSVP update.
	observer.addAction('wpem_rsvp_update', json => {
		if (+json.event_id && id == json.event_id) {
			state = json.status;

			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		}
	});

	updateRsvpButton($toggle, state, 'loading');
	rsvp(id)
		.then(json => {
			state = json.status;

			$options.on('click', function (e) {
				e.preventDefault();

				let newState = $(this).data('state');
				if (newState === state) {
					return;
				}

				// Optimistic update.
				updateRsvpButton($toggle, newState, 'loading');
				// No need to update the button here as it will be handled by the `wpem_rsvp_update` action above.
				rsvp(id, newState)
					.then(json => (state = json.status))
					.catch(error => error && alert(error));
			});
		})
		.catch(error => error && alert(error))
		.finally(() => {
			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		});
}

/**
 * Initialize RSVP notification button toggle.
 *
 * @param {Element} item
 */
function initRsvpNotifications(item) {
	let $button = $(item).find('.ps-js-btn-notifications');
	if (!$button.length) {
		return;
	}

	let id = $button.data('event-id');
	let state = false;

	updateRsvpNotificationsButton($button, state, 'loading');
	rsvpNotifications(id)
		.then(json => {
			state = json.status;

			$button.on('click', function (e) {
				e.preventDefault();

				// Optimistic update.
				updateRsvpNotificationsButton($button, !state, 'loading');
				rsvpNotifications(id, !state)
					.then(json => (state = json.status))
					.catch(error => error && alert(error))
					.finally(() => updateRsvpNotificationsButton($button, state));
			});

			$button.on('mouseenter', function (e) {
				let stateText = state ? 'enabled' : 'disabled';

				let icon = $button.data(`icon-${stateText}-hover`);
				let text = $button.data(`text-${stateText}-hover`);

				// Update button label and icon.
				$button.children('i').attr('class', icon);
				$button.children('span').text(text);
			});

			$button.on('mouseleave', function (e) {
				let stateText = state ? 'enabled' : 'disabled';

				let icon = $button.data(`icon-${stateText}`);
				let text = $button.data(`text-${stateText}`);

				// Update button label and icon.
				$button.children('i').attr('class', icon);
				$button.children('span').text(text);
			});
		})
		.catch(error => error && alert(error))
		.finally(() => updateRsvpNotificationsButton($button, state));
}

/**
 * Update RSVP button label.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateRsvpButton($toggle, state = 'no', mode) {
	let icon = 'loading' === mode ? 'gci-spinner gci-spin' : RSVP_STATES[state].icon;
	let text = RSVP_STATES[state].label;

	// Update button label and icon.
	$toggle.children('i').attr('class', `gcis ${icon}`);
	$toggle.children('span').text(text);

	// Disable button on loading.
	$toggle.css('opacity', 'loading' === mode ? 0.3 : '');
	$toggle.prop('disabled', 'loading' === mode);
}

/**
 * Update RSVP notifications button label.
 *
 * @param {JQuery} $button
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateRsvpNotificationsButton($button, state = false, mode) {
	let stateText = state ? 'enabled' : 'disabled';
	let icon = 'loading' === mode ? 'gcis gci-spinner gci-spin' : $button.data(`icon-${stateText}`);
	let text = $button.data(`text-${stateText}`);
	let aria = $button.data(`aria-${stateText}`);

	// Update button label and icon.
	$button.children('i').attr('class', icon);
	$button.children('span').text(text);
	$button.attr('aria-label', aria);

	// Disable button on loading.
	$button.css('opacity', 'loading' === mode ? 0.3 : '');
	$button.prop('disabled', 'loading' === mode);
}

$(function () {
	let item = document.querySelector('.wpem-single-event-page');
	if (item && USER_ID) {
		initItem(item);
	}
});
