import $ from 'jquery';
import { observer, template } from 'peepso';
import peepsodata from 'peepsodata';
import { rsvp } from './ajax';

const wpem = window.peepso_wpem;

const USER_ID = +peepsodata.currentuserid;
const RSVP_STATES = (wpem.data && wpem.data.rsvp_states) || {};
const RSVP_DROPDOWN = (wpem.tmpl && wpem.tmpl.rsvp_dropdown_widget) || '';

/**
 * Initialize RSVP dropdown toggle.
 *
 * @param {Element} item
 */
function initRsvp(item) {
	let $item = $(item);
	let $container = $item.find('.wpem-event-infomation');
	let $dropdown = $(template(RSVP_DROPDOWN)({ event_id: 0 })).appendTo($container);
	if (!$dropdown.length) {
		return;
	}

	let $toggle = $dropdown.find('.ps-js-dropdown-toggle');
	let $options = $dropdown.find('button[data-state]');
	let id = $dropdown.data('event-id');
	let url = $item.find('.wpem-event-action-url').attr('href');
	let state = 'no';

	// Listen to global RSVP update.
	observer.addAction('wpem_rsvp_update', json => {
		if (+json.event_id && id == json.event_id) {
			state = json.status;

			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		}
	});

	updateRsvpButton($toggle, state, 'loading');
	rsvp(url)
		.then(json => {
			id = json.event_id;
			state = json.status;

			// Update event_id acquired from the ajax response.
			$dropdown.attr('data-event-id', id);

			$options.on('click', function (e) {
				e.preventDefault();

				let newState = $(this).data('state');
				if (newState === state) {
					return;
				}

				// Optimistic update.
				updateRsvpButton($toggle, newState, 'loading');
				// No need to update the button here as it will be handled by the `wpem_rsvp_update` action above.
				rsvp(id, newState)
					.then(json => (state = json.status))
					.catch(error => error && alert(error));
			});

			updateRsvpButton($toggle, state);
			$options.removeClass('active');
			$options.filter(`[data-state="${state}"]`).addClass('active');
		})
		// Remove dropdown if error is returned on the first API call.
		.catch(() => $dropdown.remove());
}

/**
 * Update RSVP button label.
 *
 * @param {JQuery} $toggle
 * @param {string} state
 * @param {string|undefined} mode
 */
function updateRsvpButton($toggle, state = 'no', mode) {
	let icon = 'loading' === mode ? 'gci-spinner gci-spin' : RSVP_STATES[state].icon;
	let text = RSVP_STATES[state].label;

	// Update button label and icon.
	$toggle.children('i').attr('class', `gcis ${icon}`);
	$toggle.children('span').text(text);

	// Disable button on loading.
	$toggle.css('opacity', 'loading' === mode ? 0.3 : '');
	$toggle.prop('disabled', 'loading' === mode);
}

$(function () {
	let items = [...document.querySelectorAll('.wpem-single-event-widget')];
	if (items.length && USER_ID) {
		items.forEach(item => {
			initRsvp(item);
		});
	}
});
