<?php
/**
 * Plugin Name: PeepSo Integrations: WP Event Manager
 * Plugin URI: https://PeepSo.com
 * Description: Integrate event calendars into your community. Requires the WP Event Manager plugin.
 * Author: PeepSo
 * Author URI: https://www.PeepSo.com
 * Version: 7.0.1.0
 * Copyright: (c) PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later (PHP), property of PeepSo, Inc. (CSS, JavaScript, images)
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: peepso-wpem
 * Domain Path: /language
 *
 * The PHP code of this plugin is Open Source. This PHP software can be redistributed and/or modified under the terms of the 
 * GNU General Public License (version 2 or later), as published by the Free Software Foundation. 
 * See the GNU General Public License or the LICENSE file for more details.  
 *  
 * Unless explicitly stated otherwise inside the respective files, all CSS, JavaScript, and images included in this plugin are proprietary to PeepSo, Inc.  
 * They are protected by copyright and other applicable intellectual property laws. 
 * Unauthorized use, reproduction, or distribution of these assets is STRICTLY PROHIBITED.  
 *  
 * "PeepSo" and "Your Community. Your Way." are registered trademarks of PeepSo, Inc.  
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY. 
*/


class PeepSo_WPEM_Plugin {

    private static $_instance = NULL;

    const PLUGIN_NAME    = 'Integrations: WP Event Manager';
    const PLUGIN_VERSION = '7.0.1.0';
    const PLUGIN_RELEASE = ''; //ALPHA1, BETA1, RC1, '' for STABLE

    const THIRDPARTY_MIN_VERSION = '3.1.33';

    const PLUGIN_EDD = 42527003;
    const PLUGIN_SLUG = 'peepso-wpem';

    const MODULE_ID = 6663;

	private static function ready() {
        if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION) && self::ready_thirdparty()) {
            return TRUE;
        }
    }

    private static function ready_thirdparty() {
        $result = TRUE;

        if(!class_exists('WP_Event_Manager') || !version_compare( EVENT_MANAGER_VERSION, self::THIRDPARTY_MIN_VERSION, '>=' )) {
            $result = FALSE;
        }


        return $result;
    }

    public function __construct() {

        /** VERSION INDEPENDENT hooks **/

        // Admin
        add_filter('peepso_license_config', function($list){
            $data = array(
                'plugin_slug' => self::PLUGIN_SLUG,
                'plugin_name' => self::PLUGIN_NAME,
                'plugin_edd' => self::PLUGIN_EDD,
                'plugin_version' => self::PLUGIN_VERSION
            );
            $list[] = $data;
            return ($list);
        });

        if (is_admin()) {
            add_action('admin_init', array(&$this, 'dependency_check'));
        }

        // Compatibility
        add_filter('peepso_all_plugins', function($plugins){
            $plugins[plugin_basename(__FILE__)] = get_class($this);
            return $plugins;
        });

        // Translations
        add_filter('peepso_absolute_textdomain_path', function($path, $domain){
            if($domain == 'peepso-wpem') {
                return dirname(__FILE__) . DIRECTORY_SEPARATOR . 'language' . DIRECTORY_SEPARATOR;
            }
            return $path;
        },10,2);

        add_action('init', function(){
            $path = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'language' . DIRECTORY_SEPARATOR . 'peepso-wpem-' . determine_locale() . '.mo';
            load_textdomain('peepso-wpem', $path);
        });

        // Activation
        register_activation_hook(__FILE__, array(&$this, 'activate'));

        /** VERSION LOCKED hooks **/
        if (self::ready()) {

            if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) { return; }

            add_filter('peepso_admin_config_tabs', function( $tabs ) {
                $tabs['wp-events-manager'] = array(
                    'label' => __('WP Event Manager', 'peepso-wpem'),
                    'icon' => 'https://cdn.peepso.com/icons/plugins/'.self::PLUGIN_EDD.'.svg',
                    'tab' => 'wp-events-manager',
                    'function' => 'PeepSoConfigSectionWPEM',
                    'cat'   => 'integrations',
                );

                return $tabs;
            });

            add_action('peepso_init', array(&$this, 'init'));
        }
    }

    /**
     * Retrieve singleton class instance
     * @return PeepSo_WPEM_Plugin instance
     */
    public static function get_instance()
    {
        if (NULL === self::$_instance) {
            self::$_instance = new self();
        }
        return (self::$_instance);
    }

    /**
     * Check if PeepSo class is present (ie the PeepSo plugin is installed and activated)
     * If there is no PeepSo, immediately disable the plugin and display a warning
     * Check if WPEM is present, otherwise display a warning
     * Run license and new version checks against PeepSo.com
     * @return bool
     */
    public function dependency_check()
    {
        if (!class_exists('PeepSo')) {
            add_action('admin_notices', array(&$this, 'peepso_disabled_notice'));
            unset($_GET['activate']);
            deactivate_plugins(plugin_basename(__FILE__));
            return (FALSE);
        }

        if (!self::ready_thirdparty()) {

            add_action('admin_notices', function() {
                if(method_exists('PeepSo','third_party_warning')) {
                    PeepSo::third_party_warning('WP Event Manager','wp-event-manager',FALSE,self::THIRDPARTY_MIN_VERSION, self::PLUGIN_NAME);
                }
            }, 10200);
        }

        // PeepSo.com license check
        if (!PeepSoLicense::check_license(self::PLUGIN_EDD, self::PLUGIN_SLUG)) {
            add_action('admin_notices', function() {
                PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG);
            });
        }

        if (isset($_GET['page']) && 'peepso_config' == $_GET['page'] && !isset($_GET['tab'])) {
            add_action('admin_notices', function() {
                PeepSo::license_notice(self::PLUGIN_NAME, self::PLUGIN_SLUG, true);
            });
        }

        // PeepSo.com new version check
        // since 1.7.6
        if(method_exists('PeepSoLicense', 'check_updates_new')) {
            PeepSoLicense::check_updates_new(self::PLUGIN_EDD, self::PLUGIN_SLUG, self::PLUGIN_VERSION, __FILE__);
        }

        return (TRUE);
    }

    /**
     * Display a message about PeepSo not present
     */
    public function peepso_disabled_notice()
    {
        ?>
        <div class="error peepso">
            <strong>
                <?php
                echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepso-wpem'), self::PLUGIN_NAME),
                ' <a href="plugin-install.php?tab=plugin-information&amp;plugin=peepso-core&amp;TB_iframe=true&amp;width=772&amp;height=291" class="thickbox">',
                __('Get it now!', 'peepso-push'),
                '</a>';
                ?>
            </strong>
        </div>
        <?php
    }

    /**
     * Plugin activation.
     * Check PeepSo
     * Run installation
     * @return bool
     */
    public function activate()
    {
        if (!$this->dependency_check()) {
            return (FALSE);
        }

        require_once(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'install' . DIRECTORY_SEPARATOR . 'activate.php');
        $install = new PeepSO_WPEM_Install();
        $res = $install->plugin_activation();
        if (FALSE === $res) {
            // error during installation - disable
            deactivate_plugins(plugin_basename(__FILE__));
        }

        return (TRUE);
    }

    public function init()
    {
        PeepSo::add_autoload_directory(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR);
        PeepSoTemplate::add_template_directory(plugin_dir_path(__FILE__));

        // PeepSo Navigation Profile
        add_filter('peepso_navigation_profile', function($links){

            // do nothing if the option is disabled
            if(0==PeepSo::get_option_new('wpem_profiles_enable')) {
                return $links;
            }

            // do nothing if "owner only" is enabled and we are loading someone elses profile
            if(1==PeepSo::get_option('wpem_profiles_owner_only',0)) {
                if(isset($links['_user_id']) && get_current_user_id() != $links['_user_id']) {
                    return $links;
                }
            }

            $slug = PeepSo::get_option_new('wpem_profiles_slug');
            $links[$slug] = array(
                'href' => $slug,
                'label'=> PeepSo::get_option_new('wpem_profiles_label'),
                'icon' => 'gcis gci-calendar',
            );

            return $links;
        });

        // PeepSo Navigation
        add_filter('peepso_navigation', function ($navigation) {

            if($page_id = get_option('event_manager_events_page_id')) {

                $navigation['events'] = array(
                    'href' => get_the_permalink($page_id),
                    'label' => get_the_title($page_id),
                    'icon' => 'gcis gci-calendar',

                    'primary' => TRUE,
                    'secondary' => FALSE,
                    'mobile-primary' => TRUE,
                    'mobile-secondary' => FALSE,
                    'widget' => TRUE,
                );

            }

            return $navigation;
        });

        if (is_admin()) {
            add_action('admin_enqueue_scripts', function(){
                wp_enqueue_script('peepso-admin-config-wpem',
                    PeepSo::get_asset('js/admin/config-wpem.js', __FILE__),
                    array('jquery'), self::PLUGIN_VERSION, TRUE);
            });
        } else {
            add_filter('has_wpem_shortcode', '__return_true');

            add_action('wp_enqueue_scripts', function() {
                wp_enqueue_script('peepso-wpem',
                    PeepSo::get_asset('js/bundle.min.js', __FILE__),
                    array('peepso'), self::PLUGIN_VERSION, TRUE);

                wp_register_style('peepso-wpem-custom-css', plugin_dir_url(__FILE__) . 'assets/css/custom.css', self::PLUGIN_VERSION, 'all');
                wp_enqueue_style('peepso-wpem-custom-css');

                $states = PeepSoWPEM_RSVP::states();
                wp_localize_script('peepso-wpem', 'peepso_wpem', [
                    'tmpl' => [
                        'rsvp_dropdown_widget' => PeepSoTemplate::exec_template('rsvp', 'dropdown-widget', NULL, TRUE),
                        'rsvp_dropdown_stream' => PeepSoTemplate::exec_template('rsvp', 'dropdown-stream', NULL, TRUE)
                    ],
                    'data' => [
                        'rsvp_states' => $states
                    ]
                ]);
            });

            /** Profile view filters **/

            // Filter AJAX calls (default / free plugin views)

            add_filter('get_event_listings_cache_results', function($cache) {
                if(self::form_data_peepso_user_id()) {
                    return FALSE;
                }

                return $cache;
            });

            add_filter('event_manager_get_listings', function($args) {

                if(  ('mine') == self::form_data_peepso_mode() && $peepso_user_id = self::form_data_peepso_user_id()) {
                    $args['author__in'] = $peepso_user_id;
                }

                return $args;
            });

            // PeepSo/PeepSo#5764 filter result for wpem-calendar
            add_action('pre_get_posts', function($query) {

                if(  ('mine') == self::form_data_peepso_mode() && $peepso_user_id = self::form_data_peepso_user_id()) {
                    $query->set('author__in', $peepso_user_id);
                }

                return $query;
            });

            add_filter('posts_join', function($join, $wp_query) {

                $mode = self::form_data_peepso_mode();

                if( (stristr($mode,'rsvp') || $mode == 'everything') && $peepso_user_id = self::form_data_peepso_user_id()) {
                    global $wpdb;
                    $join .= " LEFT JOIN {$wpdb->prefix}peepso_wpem_rsvp as rsvp ON rsvp.event_id={$wpdb->posts}.ID ";
                }

                return $join;
            },10,2);

            add_filter('posts_where', function($where, $wp_query){
                $mode = self::form_data_peepso_mode();

                if( (stristr($mode,'rsvp') || $mode == 'everything') && $peepso_user_id = self::form_data_peepso_user_id()) {
                    global $wpdb;
                    if($mode == 'rsvp-yes') { $in="'yes'"; }
                    if($mode == 'rsvp-maybe') { $in="'maybe'"; }
                    if($mode == 'rsvp-any' || $mode == 'everything') {
                        $in="'yes','maybe'";
                    }

                    $q_everything =  '';
                    if($mode == 'everything') {
                        $q_everything = "({$wpdb->posts}.post_author = $peepso_user_id AND {$wpdb->posts}.post_author = rsvp.user_id) OR ";
                    }
                    $where .= " AND ( $q_everything (rsvp.user_id = $peepso_user_id AND rsvp.status IN ($in) AND rsvp.user_id IS NOT NULL)) ";
                }

                return $where;
            }, 10,2);

// Filter AJAX calls (calendar / paid)
// @TODO
//        // WPEM Paid: filter by user / [events_calendar] shortcode
//        add_action('pre_get_posts', function($query) {
//            if($query instanceof WP_Query && isset($query->query) && is_array($query->query)) {
//                if(isset($query->query['post_type']) && $query->query['post_type'] == 'event_listing') {
//                    if(!PeepSo::is_api_request()) {
//                        $PeepSoUrlSegments = new PeepSoUrlSegments();
//                        if('peepso_profile' == $PeepSoUrlSegments->_shortcode) {
//                            $query->query_vars['author__in'] = $query->query['author__in'] = [PeepSoProfileShortcode::get_instance()->get_view_user_id()];
//                        }
//                    }
//                }
//            }
//
//            return $query;
//        });

            /** Wrapper integration **/

            // Wrapper
            if(PeepSo::get_option_new('wpem_peepso_wrapper_enable')) {
                add_filter('the_content', function ($content) {

                    if (is_embed()) {
                        return $content;
                    }

                    global $post;

                    $do_wrapper = FALSE;

                    if ($post instanceof WP_Post && $id = $post->ID) {

                        if ('event_listing' == $post->post_type) {
                            $do_wrapper = TRUE;
                        }
                        if (!$do_wrapper) {
                            $page_ids = [
                                get_option('event_manager_events_page_id'),
                                get_option('event_manager_submit_event_form_page_id'),
                                get_option('event_manager_event_dashboard_page_id'),
                                get_option('event_manager_organizer_dashboard_page_id')
                            ];

                            if (in_array($id, $page_ids)) {
                                $do_wrapper = TRUE;
                            }
                        }
                    }

                    if ($do_wrapper) {
                        return PeepSoTemplate::exec_template('wpem', 'wrapper', ['content' => $content], TRUE);
                    }

                    return $content;
                });

            }

            // Organizer name - organizers enabled
            add_action('display_organizer_name', function($name, $post) {

                //$option = PeepSo::get_option_new('wpem_organizer_override');
                //if( ( 2 == $option ) || (1==$option && !strlen($name)) ) {
                if(!strlen($name)) {
                    return PeepSoTemplate::exec_template('organizer','name',['post'=>$post],TRUE);
                }

                return $name;

            },2,9999);

            // Organizer box
            add_action('single_event_sidebar_start', function(){

                $space = FALSE;

                if(PeepSo::get_option_new('wpem_rsvp_enable')) {
                    PeepSoTemplate::exec_template('rsvp', 'notifications', ['space'=>$space]);
                    $space = TRUE;
                }

                if(PeepSo::get_option_new('wpem_organizer_side')) {
                    PeepSoTemplate::exec_template('organizer', 'side',['space'=>$space]);
                    $space = TRUE;
                }

                if(PeepSo::get_option_new('wpem_rsvp_enable') && PeepSo::get_option_new('wpem_rsvp_sidebar_attendees') > 0) {
                    PeepSoTemplate::exec_template('attendees', 'side', ['space' => $space]);
                }
            });


            /** Profile integration **/
            $profile_slug = PeepSo::get_option_new('wpem_profiles_slug');

            add_action('peepso_profile_segment_' . $profile_slug, function () {

                $view_user_id = PeepSoUrlSegments::get_view_id(PeepSoProfileShortcode::get_instance()->get_view_user_id());

                // Verify if the option is enabled and the user has the right to come here.
                $continue = TRUE;

                // If tabs are disabled
                if(0==PeepSo::get_option_new('wpem_profiles_enable')) { $continue = FALSE; }

                // Or "Owner Only" is enabled and we are not the owner
                if(1==PeepSo::get_option_new('wpem_profiles_owner_only')) {
                    if(get_current_user_id() !=  $view_user_id) {
                        $continue = FALSE;
                    }
                }

                if($continue) {
                    // If everything is OK, print the HTML
                    $template = 'profile-wpem-main';
                    $PeepSoURLSegments = new PeepSoUrlSegments();

                    if (in_array($segment = $PeepSoURLSegments->get(3), ['create', 'manage'])) {
                        if(PeepSo::get_option_new('wpem_fem_enable')) {
                            $template = 'profile-wpem-' . $segment;
                        }
                    }

                    // if()... theoretical other sub-tabs

                    echo PeepSoTemplate::exec_template('wpem', $template, array('view_user_id' => $view_user_id), TRUE);
                } else {
                    // If not, redirect gracefully to profile home
                    PeepSo::redirect(PeepSoUser::get_instance($view_user_id)->get_profileurl());
                }
            });

            /** Hijack dashboard link **/
            add_filter('page_link', function($link, $post_id, $sample){

                if(!get_current_user_id()) { return $link; }                                            // Do nothing for guests
                if(!PeepSo::get_option_new('wpem_fem_enable')) { return $link; }                  // Do nothing if FEM is disabled

                // Dashboard
                if($post_id == get_option('event_manager_event_dashboard_page_id'))  {
                    return PeepSo_WPEM_Plugin::get_url(get_current_user_id(),'manage');
                }

                // Submit
                if($post_id == get_option('event_manager_submit_event_form_page_id'))  {
                    return PeepSo_WPEM_Plugin::get_url(get_current_user_id(),'create');
                }

                return $link;

            },99,3);

            /** Fire WPEM action handlers in profile tabs **/
            add_action('template_redirect', function() {

                if(!PeepSo::get_option_new('wpem_fem_enable')) { return; }                  // Do nothing if FEM is disabled

                global $post;

                if ($post instanceof WP_Post && stristr($post->post_content, '[peepso_profile')) {
                    $wpem = new WP_Event_Manager_Shortcodes();
                    $wpem->event_dashboard_handler();
                    $wpem->organizer_dashboard_handler();
                    $wpem->venue_dashboard_handler();
                }
            });

            if (PeepSo::get_option_new('wpem_fem_enable') && strpos($_SERVER['REQUEST_URI'], PeepSo::get_option_new('wpem_profiles_slug') . '/manage') !== FALSE) {

                // these 2 filters below are added to override event_manager_get_permalink result
                add_filter('pre_option_event_manager_submit_event_form_page_id', function($option) {
                    if (isset($_GET['action']) && $_GET['action'] == 'duplicate') {
                        $GLOBALS['peepso_wpem_submit_event_form'] = true;
                        // override the id to anything else, and we can't use negative value because there's absint check in event_manager_get_page_id to force value to be absolute or using any random id because there's a check to make sure if the id is exits
                        return $GLOBALS['post']->ID;
                    }
                    return $option;
                });

                add_filter('post_link', function($link, $post) {
                    if ($post->ID == $GLOBALS['post']->ID && isset($GLOBALS['peepso_wpem_submit_event_form'])) {
                        return PeepSo_WPEM_Plugin::get_url(get_current_user_id(),'create');
                    }
                    return $link;
                }, 10, 2);

                // Override buttons
                if (!PeepSo::get_option_new('disable_questionmark_urls')) {
                    add_action( 'template_redirect', function() {
                        ob_start(function($buffer) {
                            $replacement = PeepSo_WPEM_Plugin::get_url(get_current_user_id(),'manage').'&action=';

                            $pattern = '/(<div class="wpem-dboard-event-act-btn"><a href=")([^"]*[^"]*)("?)/';
                            $buffer = preg_replace_callback($pattern, function($matches) use ($replacement) {
                                if (strpos($matches[2], '?action=')) {
                                    $url_part = explode('?action=', $matches[2]);
                                    $new_replacement = $replacement . $url_part[1];
                                    return $matches[1] . $new_replacement . $matches[3];
                                }
                                return $matches[1] . $matches[2] . $matches[3];
                            }, $buffer);

                            return $buffer;
                        });
                    }, 99);
                }

                // Override form action attribute on edit event form
                if (!empty($_GET['action']) && $_GET['action'] == 'edit' && !empty($_GET['event_id'])) {
                    add_action( 'template_redirect', function() {
                        ob_start(function($buffer) {
                            $action_url = add_query_arg(
                                [
                                    'action' => 'edit',
                                    'event_id' => intval($_GET['event_id'])
                                ],
                                PeepSo_WPEM_Plugin::get_url(get_current_user_id(), 'manage')
                            );
                            
                            $pattern = '/(<form[^>]*action=")[^"]*("[^>]*id="submit-event-form"[^>]*>)/i';
                            $replacement = '$1' . $action_url . '$2';
                            $buffer = preg_replace($pattern, $replacement, $buffer);
                            
                            return $buffer;
                        } );
                    }, 99);
                }
            }

            /** Permissions */
            PeepSoWPEM_Permissions::get_instance();
        }

        /** RSVP **/
        if(PeepSo::get_option_new('wpem_rsvp_enable')) {

            PeepSoWPEM_RSVP::get_instance();

            if(!PeepSo::get_option_new('wpem_display_event_registration_method')) {
                add_filter('get_event_registration_method', '__return_false',99,2);
                add_filter('display_event_registration_method', '__return_false',99,2);
            }
        }

// Always mark event owner as "going", even if RSVP is currently disabled
// Handle the default notification state
        add_action('publish_event_listing', function($ID, $post) {
            if($post->post_type = 'event_listing') {
                // Default notification flag
                $default = 1; // TBD configurable
                update_post_meta($ID,'peepso_wpem_rsvp_notifications', $default);

                // Default RSVP status
                $default = 'yes'; // TBD configurable
                $RSVP =  new PeepSOWPEM_RSVP_User($ID, $post->post_author);
                $RSVP->rsvp($default);
            }
        },10,3);

        /** Activity Stream **/
        PeepSoWPEM_Activity::get_instance();

        /** Notifications **/
        PeepSoWPEM_Notifications::get_instance();

        /** REST API  **/
        add_filter('peepso_rest_paths', function($rest_paths) {
            $rest_paths[] = dirname(__FILE__)  . DIRECTORY_SEPARATOR . 'api'. DIRECTORY_SEPARATOR . 'rest' . DIRECTORY_SEPARATOR . PeepSo3_API::REST_V;
            return $rest_paths;
        });
    }

    /** Utilities **/

    public static function check_post_type($post) {
        return ($post instanceof  WP_Post && $post->post_type == 'event_listing');
    }

    public static function get_url(int $view_id = 0, $page='')
    {
        if(!$view_id) {
            $view_id = get_current_user_id();
        }

        $user = PeepSoUser::get_instance($view_id);

        switch($page) {
            case 'create':
                return PeepSo_WPEM_Plugin::get_url($view_id).'/create';
                break;
            case 'manage':
                return PeepSo_WPEM_Plugin::get_url($view_id).'/manage';
                break;
            default:
                return $user->get_profileurl(). PeepSo::get_option_new('wpem_profiles_slug');
        }
    }

    // Parse form_data sent by the WPEM AJAX search, utilized in profile view
    private static function form_data() {

        if(isset($_REQUEST['form_data'])) {
            $form_data = FALSE;
            parse_str($_REQUEST['form_data'], $form_data);
            return $form_data;
        }

        return FALSE;
    }

    private static function form_data_peepso_user_id() {

        if (is_array($form_data = self::form_data()) && array_key_exists('wpem_peepso_user_id', $form_data)) {
            return $form_data['wpem_peepso_user_id'];
        } else if (isset($_POST['wpem_peepso_user_id'])) {
            return $_POST['wpem_peepso_user_id'];
        }

        return FALSE;
    }

    private static function form_data_peepso_mode() {
        if (is_array($form_data = self::form_data()) && array_key_exists('wpem_peepso_mode', $form_data)) {
            return $form_data['wpem_peepso_mode'];
        } else if (isset($_POST['wpem_peepso_mode'])) {
            return $_POST['wpem_peepso_mode'];
        }

        return self::form_data_peepso_user_id() ? 'mine' : FALSE;
    }
}

PeepSo_WPEM_Plugin::get_instance();
